# RevPlan OData Architecture Documentation

## Overview

This document outlines the architecture of the RevPlan OData middleware layer, which provides a clean interface between client applications and FileMaker OData services. The application uses a hardcoded approach for database, table, and field configurations to ensure type safety, maintainability, and clarity.

## Core Architecture

The application follows a layered architecture:

1. **Configuration Layer**: Defines database structures, tables, and fields
2. **API Layer**: Handles OData requests and communicates with FileMaker
3. **Context Layer**: Provides React context for database state management
4. **Hooks Layer**: Offers React hooks for component-level data access

## Hardcoded Approach

### Motivation

The hardcoded approach replaces a previous implementation that used virtual/dynamic tables and fields. The benefits include:

- **Type Safety**: TypeScript can validate field names and types at compile time
- **IDE Support**: Better autocomplete and refactoring support
- **Maintainability**: Easier to track changes and understand the data model
- **Performance**: No runtime overhead for resolving dynamic field names
- **Reliability**: Reduced risk of runtime errors due to typos or missing fields

### Implementation

#### Configuration Files

- **`/lib/config/field-names.ts`**: Contains actual database field names as constants
- **`/lib/config/databases.ts`**: Defines the database structure with tables and fields

```typescript
// Example from databases.ts
export const hosts = {
  host1: {
    hostId: 'host1',
    baseUrl: 'https://db-data.net',
    apiPath: '/fmi/odata/v4',
    authType: 'basic',
    databases: {
      'TrackPad-FMS-test': {
        tables: {
          TrackPad: {
            displayName: 'Track Pad',
            fields: trackPadFields
          },
          Clients: {
            displayName: 'Clients',
            fields: clientFields
          }
        }
      }
    }
  }
};
```

#### OData Client

The `ODataClient` class (`/lib/api/odataClient.ts`) handles communication with the FileMaker OData service:

- Authentication with FileMaker OData endpoints
- CRUD operations (Create, Read, Update, Delete)
- Error handling and response formatting
- Script execution forwarding

#### API Routes

The Next.js API routes (`/app/api/odata/[...path]/route.ts`) provide a RESTful interface:

- Dynamic routing for different databases, tables, and records
- Request validation and parameter extraction
- Error handling and response formatting
- Script execution forwarding

#### React Integration

- **`DatabaseContext`** (`/lib/context/DatabaseContext.tsx`): Manages current host, database, and table selections
- **`useOData`** hook (`/lib/hooks/useOData.ts`): Provides data fetching and CRUD operations for React components

## Authentication

The application uses Basic Authentication with FileMaker OData services. Credentials are stored in environment variables:

- `FILEMAKER_ODATA_URL1`: Base URL for the FileMaker OData service
- `FILEMAKER_USERNAME1`: Username for authentication
- `FILEMAKER_PASSWORD1`: Password for authentication

## API Endpoints

### Main Endpoints

- **GET** `/api/odata/{databaseName}/{tableName}`: List records
- **GET** `/api/odata/{databaseName}/{tableName}/{id}`: Get a single record
- **POST** `/api/odata/{databaseName}/{tableName}`: Create a record
- **PATCH** `/api/odata/{databaseName}/{tableName}/{id}`: Update a record
- **DELETE** `/api/odata/{databaseName}/{tableName}/{id}`: Delete a record

### Script Execution

- **GET** `/api/odata/{databaseName}/Script.{scriptName}`: Execute a FileMaker script (GET)
- **POST** `/api/odata/{databaseName}/Script.{scriptName}`: Execute a FileMaker script (POST)

## Query Parameters

The OData client supports standard OData query parameters:

- `$select`: Field selection
- `$filter`: Filtering
- `$orderby`: Sorting
- `$top`/`$skip`: Pagination
- `$count`: Record counting

## Adding New Tables or Fields

To add new tables or fields to the application:

1. Add the field names to `/lib/config/field-names.ts`
2. Update the database configuration in `/lib/config/databases.ts`
3. No changes to the OData client or API routes are required

## Best Practices

1. **Always use typed field names** from the configuration files
2. **Never hardcode field names** directly in components
3. **Use the `useOData` hook** for data fetching in React components
4. **Handle loading and error states** properly in the UI
5. **Use the DatabaseContext** for managing current selections

## Known Limitations

1. Next.js App Router warning about `params.path` in dynamic routes
   - This is a known issue with the Next.js App Router and doesn't affect functionality
   - The warning occurs because Next.js expects dynamic route parameters to be handled asynchronously

## Future Improvements

1. Add comprehensive test coverage
2. Implement caching for frequently accessed data
3. Add pagination controls in the UI
4. Enhance error handling and user feedback
5. Add more robust type definitions for API responses
