'use client';

import { useState, useEffect, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import DatabaseSelector from '@/components/DatabaseSelector';
import NavBar from '@/components/NavBar';

interface FormField {
  key: string;
  value: string;
  type: string;
  required: boolean;
  displayName: string;
  multiline?: boolean;
}

// Main content component that uses useSearchParams
function NewClientContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { 
    currentHostId, 
    currentDatabaseName, 
    setCurrentHostId,
    setCurrentDatabaseName,
    setCurrentTableName
  } = useDatabaseContext();
  
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [saveSuccess, setSaveSuccess] = useState(false);
  
  // Form state
  const [formData, setFormData] = useState({
    Name: '',
    Category: '',
    Company: ''
  });
  
  // Get parameters from URL
  const hostId = searchParams ? searchParams.get('host') : null;
  const databaseName = searchParams ? searchParams.get('database') : null;
  
  // Set the context values from URL parameters if provided
  useEffect(() => {
    if (hostId) setCurrentHostId(hostId);
    if (databaseName) setCurrentDatabaseName(databaseName);
    setCurrentTableName('Clients'); // Default table for clients
  }, [hostId, databaseName, setCurrentHostId, setCurrentDatabaseName, setCurrentTableName]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!currentHostId || !currentDatabaseName) {
      setError('Please select a host and database');
      return;
    }

    // Simple validation
    if (!formData.Name.trim()) {
      setError('Name is required');
      return;
    }

    try {
      setSaving(true);
      setError(null);
      
      // Match the working Postman request format
      const requestData = {
        Name: formData.Name.trim(),
        Company: formData.Company.trim() || ''
        // Note: Removed Category field as it's not in the working Postman example
      };
      
      console.log('Submitting client data:', requestData);
      
      const response = await fetch(`/api/odata/${currentDatabaseName}/Clients`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Host-Id': currentHostId,
          'Accept': 'application/json',
          'OData-Version': '4.0',
          'OData-MaxVersion': '4.0'
        },
        body: JSON.stringify(requestData)
      });
      
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || `Failed to create client: ${response.statusText}`);
      }
      
      setSaveSuccess(true);
      // Redirect to clients-all page after a short delay
      setTimeout(() => {
        router.push('/clients-all');
      }, 1500);
      
    } catch (err) {
      console.error('Error creating client:', err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred');
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex justify-center items-center h-64">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900"></div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <NavBar />
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="px-4 py-6 sm:px-0">
          <div className="flex justify-between items-center mb-6">
            <h1 className="text-2xl font-semibold text-gray-900">New Client</h1>
            <Link 
              href="/client" 
              className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
            >
              Back to Clients
            </Link>
          </div>
          
          {error && (
            <div className="mb-4 bg-red-50 border-l-4 border-red-400 p-4">
              <div className="flex">
                <div className="flex-shrink-0">
                  <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                  </svg>
                </div>
                <div className="ml-3">
                  <p className="text-sm text-red-700">{error}</p>
                </div>
              </div>
            </div>
          )}
          
          {saveSuccess && (
            <div className="mb-4 bg-green-50 border-l-4 border-green-400 p-4">
              <div className="flex">
                <div className="flex-shrink-0">
                  <svg className="h-5 w-5 text-green-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                  </svg>
                </div>
                <div className="ml-3">
                  <p className="text-sm text-green-700">Client created successfully! Redirecting...</p>
                </div>
              </div>
            </div>
          )}
          
          <div className="bg-white shadow overflow-hidden sm:rounded-lg">
            <form onSubmit={handleSubmit} className="space-y-6 p-6">
              <div className="space-y-4">
                {/* Name Field */}
                <div>
                  <label htmlFor="Name" className="block text-sm font-medium text-gray-700">
                    Name <span className="text-red-500">*</span>
                  </label>
                  <div className="mt-1">
                    <input
                      type="text"
                      name="Name"
                      id="Name"
                      required
                      value={formData.Name}
                      onChange={handleInputChange}
                      className="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md"
                      placeholder="Enter client name"
                    />
                  </div>
                </div>
                
                {/* Company Field */}
                <div>
                  <label htmlFor="Company" className="block text-sm font-medium text-gray-700">
                    Company
                  </label>
                  <div className="mt-1">
                    <input
                      type="text"
                      name="Company"
                      id="Company"
                      value={formData.Company}
                      onChange={handleInputChange}
                      className="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md"
                      placeholder="Enter company name"
                    />
                  </div>
                </div>
                
                {/* Category Field */}
                <div>
                  <label htmlFor="Category" className="block text-sm font-medium text-gray-700">
                    Category
                  </label>
                  <div className="mt-1">
                    <input
                      type="text"
                      name="Category"
                      id="Category"
                      value={formData.Category}
                      onChange={handleInputChange}
                      className="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md"
                      placeholder="Enter category"
                    />
                  </div>
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200">
                <Link
                  href="/client"
                  className="bg-white py-2 px-4 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                >
                  Cancel
                </Link>
                <button
                  type="submit"
                  disabled={saving}
                  className={`inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 ${saving ? 'opacity-70 cursor-not-allowed' : ''}`}
                >
                  {saving ? 'Saving...' : 'Save Client'}
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function NewClientPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">New Client</h1>
              <p>Loading form...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <NewClientContent />
    </Suspense>
  );
}
