/**
 * FileMaker OData Configuration
 * Hardcoded configuration for tables and fields
 */

import { 
  clientsFields, 
  trackPadFields,
  clientsImportantFields,
  trackPadImportantFields,
  clientsEditableFields,
  trackPadEditableFields,
  trackPadDisplayShort,
  trackPadDisplayAll,
  trackPadDisplayFull,
  clientsDisplayShort,
  clientsDisplayFull
} from './field-names';

export interface FieldConfig {
  fieldName: string;
  displayName: string;
  type: 'text' | 'number' | 'date' | 'boolean' | 'select';
  options?: string[];
  required?: boolean;
  hidden?: boolean;
  readOnly?: boolean;
  multiline?: boolean;
}

export interface TableConfig {
  tableName: string;
  displayName: string;
  endpoint: string;
  idField: string;
  defaultSortField: string;
  defaultSortDirection: 'asc' | 'desc';
  fields: FieldConfig[];
  preferredColumnOrder?: string[];
  editableFields?: string[];
  readOnlyFields?: string[];
  hiddenFields?: string[];
  displayShort?: string[];
  displayFull?: string[];
}

export interface DatabaseConfig {
  databaseName: string;
  displayName: string;
  tables: TableConfig[];
}

export interface HostConfig {
  hostId: string;
  displayName: string;
  baseUrl: string;
  apiPath: string;
  authType: 'basic' | 'oauth' | 'none';
  credentials?: {
    username?: string;
    password?: string;
    clientId?: string;
    clientSecret?: string;
  };
  odataUrl?: string; // Full OData URL for direct access
  databases: DatabaseConfig[];
}

// Define the FileMaker OData host with hardcoded configurations
export const hosts: HostConfig[] = [
  {
    hostId: 'host1',
    displayName: 'FileMaker Host',
    baseUrl: 'https://db-data.net',
    apiPath: '/fmi/odata/v4',
    authType: 'basic',
    credentials: {
      username: process.env.FILEMAKER_USERNAME1 || '',
      password: process.env.FILEMAKER_PASSWORD1 || '',
    },
    odataUrl: process.env.FILEMAKER_ODATA_URL1 || 'https://db-data.net/fmi/odata/v4/TrackPad-FMS-test',
    databases: [
      {
        databaseName: 'TrackPad-FMS-test',
        displayName: 'FileMaker Database',
        tables: [
          {
            tableName: 'TrackPad',
            displayName: 'TrackPad',
            endpoint: 'TrackPad',
            idField: 'z_recid',
            defaultSortField: 'StartDate',
            defaultSortDirection: 'desc',
            editableFields: trackPadEditableFields,
            readOnlyFields: ['z_recid', 'DateCreated'],
            hiddenFields: ['proj_id', 'tsk_id', 'LenTime', 'Message_xp'],
            displayShort: trackPadDisplayAll,
            displayFull: trackPadDisplayFull,  // Add fields you want to hide
            fields: [
              { fieldName: 'z_recid', displayName: 'Record ID', type: 'number', readOnly: true },
              { fieldName: 'Name', displayName: 'Name', type: 'text' },
              { fieldName: 'StartDate', displayName: 'Start Date', type: 'date' },
              { fieldName: 'StartTime', displayName: 'Start Time', type: 'text' },
              { fieldName: 'EndTime', displayName: 'End Time', type: 'text' },
              { fieldName: 'LenTime', displayName: 'Duration', type: 'number' },
              { fieldName: 'Message', displayName: 'Message', type: 'text', multiline: true },
              { 
                fieldName: 'NoteType', 
                displayName: 'Type', 
                type: 'select', 
                options: ['GoEasy', 'Event', 'Task', 'Note']
              },
              { 
                fieldName: 'pl_Status', 
                displayName: 'Status', 
                type: 'select', 
                options: ['Active', 'Completed', 'Cancelled', 'On Hold']
              },
              { fieldName: 'pl_Cat', displayName: 'Category', type: 'text' },
              { fieldName: 'proj_id', displayName: 'Project ID', type: 'text' },
              { fieldName: 'tsk_id', displayName: 'Task ID', type: 'text' },
              { fieldName: 'DateCreated', displayName: 'Date Created', type: 'date', readOnly: true },
              { fieldName: 'DateDue', displayName: 'Due Date', type: 'date' },
            ],
            preferredColumnOrder: [
              'z_recid',
              'Name',
              'StartDate',
              'StartTime',
              'EndTime',
              'LenTime',
              'Message',
              'NoteType',
              'pl_Status',
              'pl_Cat',
              'proj_id',
              'tsk_id'
            ]
          },
          {
            tableName: 'Clients',
            displayName: 'Clients',
            endpoint: 'Clients',
            idField: 'z_recid',
            defaultSortField: 'Name',
            defaultSortDirection: 'asc',
            editableFields: clientsEditableFields,
            readOnlyFields: ['z_recid'],
            displayShort: clientsDisplayShort,
            displayFull: clientsDisplayFull,
            fields: [
              { fieldName: 'z_recid', displayName: 'Record ID', type: 'number', readOnly: true },
              { fieldName: 'Name_id', displayName: 'Name ID', type: 'number' },
              { fieldName: 'Name', displayName: 'Name', type: 'text', required: true },
              { fieldName: 'Email', displayName: 'Email', type: 'text' },
              { fieldName: 'Phone', displayName: 'Phone', type: 'text' },
              { fieldName: 'Company', displayName: 'Company', type: 'text' },
              { fieldName: 'Notes', displayName: 'Notes', type: 'text', multiline: true },
              { fieldName: 'Status', displayName: 'Status', type: 'text' },
            ],
            preferredColumnOrder: [
              'z_recid',
              'Name',
              'Email',
              'Phone',
              'Company',
              'Notes',
              'Status'
            ]
          }
        ]
      }
    ]
  }
];

// Helper functions to access configuration
export function getHost(hostId?: string): HostConfig | undefined {
  return hosts[0]; // Always return the first host
}

export function getDatabase(hostId: string, databaseName: string): DatabaseConfig | undefined {
  const host = getHost();
  return host?.databases.find(db => db.databaseName === databaseName);
}

export function getTable(hostId: string, databaseName: string, tableName: string): TableConfig | undefined {
  const database = getDatabase(hostId, databaseName);
  return database?.tables.find(table => table.tableName === tableName);
}

export function getAllTables(): { hostId: string; databaseName: string; table: TableConfig }[] {
  const result: { hostId: string; databaseName: string; table: TableConfig }[] = [];
  
  const host = hosts[0];
  for (const database of host.databases) {
    for (const table of database.tables) {
      result.push({
        hostId: host.hostId,
        databaseName: database.databaseName,
        table: table
      });
    }
  }
  
  return result;
}

// Default configuration
export const defaultHostId = 'host1';
export const defaultDatabaseName = 'TrackPad-FMS-test';
export const defaultTableName = 'TrackPad';
