'use client';

import React, { useState, useEffect } from 'react';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import NavBar from '@/components/NavBar';

interface Record {
  [key: string]: any;
}

// Helper function to get record ID from a record object
function getRecordId(record: Record): string {
  if (!record) return '';
  
  // Check @odata.id field first (standard OData field)
  if (record['@odata.id']) {
    const match = record['@odata.id'].match(/\((\d+)\)/);
    if (match && match[1]) {
      return match[1];
    }
  }
  
  // Check @id field (some OData implementations use this)
  if (record['@id']) {
    const match = record['@id'].match(/\((\d+)\)/);
    if (match && match[1]) {
      return match[1];
    }
  }
  
  // Fallback to other ID fields
  const possibleIdFields = [
    'ID', 'Id', 'id', 'z_recid', 'recid', '_id',
    'RecordID', 'record_id', 'Nid', 'nid', 'ctid'
  ];
  
  for (const idField of possibleIdFields) {
    if (record[idField] !== undefined && record[idField] !== null) {
      return String(record[idField]);
    }
  }
  
  return '';
}

// Helper function to get client ID from a task record
function getClientId(record: Record): string {
  if (!record) return '';
  
  // Check for Name_id field which contains the client ID
  if (record['Name_id'] !== undefined && record['Name_id'] !== null) {
    return String(record['Name_id']);
  }
  
  return '';
}

export default function ListPage() {
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentTableName
  } = useDatabaseContext();
  
  // Check if mobile view
  const [isMobile, setIsMobile] = useState(false);
  
  // Update mobile state on window resize and on mount
  useEffect(() => {
    // Only run on client-side
    if (typeof window === 'undefined') return;
    
    const checkIfMobile = () => {
      const isMobileView = window.innerWidth <= 768;
      console.log('Is mobile view:', isMobileView, 'Window width:', window.innerWidth);
      setIsMobile(isMobileView);
    };
    
    // Initial check
    checkIfMobile();
    
    // Add event listener for window resize
    window.addEventListener('resize', checkIfMobile);
    
    // Cleanup
    return () => window.removeEventListener('resize', checkIfMobile);
  }, []);
  
  // Mobile field labels mapping
  const mobileLabels: {[key: string]: string} = {
    'StartDate': 'Date',
    'pl_Status': 'Status',
    'Message': 'Note',
    'Name': 'Task',
    'z_recid': 'ID',
    'ClientName': 'Client'
  };
  
  // Mobile field order
  const mobileFields = ['Name', 'ClientName', 'StartDate', 'pl_Status', 'Message', 'z_recid'];
  
  // Always set table to 'TrackPad' when component mounts
  useEffect(() => {
    // Force table to be 'TrackPad' for this specialized view
    console.log('Setting table to TrackPad. Current host/db:', { currentHostId, currentDatabaseName });
    setCurrentTableName('TrackPad');
  }, [setCurrentTableName, currentHostId, currentDatabaseName]);
  
  // Debug current context values
  useEffect(() => {
    console.log('Current database context values:', {
      host: currentHostId,
      database: currentDatabaseName,
      table: currentTableName,
      tableConfig: currentTable
    });
  }, [currentHostId, currentDatabaseName, currentTableName, currentTable]);
  
  const [records, setRecords] = useState<Record[]>([]);
  const [sortedRecords, setSortedRecords] = useState<Record[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [queryParams, setQueryParams] = useState('');
  const [queryInput, setQueryInput] = useState('');
  const [searchField, setSearchField] = useState('Name');
  const [searchOperation, setSearchOperation] = useState('contains');
  
  // Sorting state - default to StartDate descending (newest first)
  const [sortField, setSortField] = useState<string>('StartDate');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('desc');
  
  // Refresh trigger for manual refreshes
  const [refreshTrigger, setRefreshTrigger] = useState<number>(0);
  
  // Handle sorting of columns
  const handleSort = (field: string) => {
    // If clicking the same field, toggle direction
    if (sortField === field) {
      setSortDirection(sortDirection === 'asc' ? 'desc' : 'asc');
    } else {
      // If clicking a new field, set it as sort field with ascending direction
      setSortField(field);
      setSortDirection('asc');
    }
  };
  
  // Apply client-side sorting when sort parameters change
  useEffect(() => {
    if (!sortField || records.length === 0) {
      // If no sort field is specified, use the original records
      setSortedRecords([...records]);
      return;
    }
    
    // Create a copy of records to sort
    const recordsCopy = [...records];
    
    // Sort the records based on the selected field and direction
    recordsCopy.sort((a, b) => {
      // Get values, handling undefined/null values
      const aValue = a[sortField] || '';
      const bValue = b[sortField] || '';
      
      // Handle different data types
      if (sortField === 'StartDate') {
        // Date comparison
        const dateA = aValue ? new Date(aValue).getTime() : 0;
        const dateB = bValue ? new Date(bValue).getTime() : 0;
        return sortDirection === 'asc' ? dateA - dateB : dateB - dateA;
      } else {
        // String comparison (case-insensitive)
        const strA = String(aValue).toLowerCase();
        const strB = String(bValue).toLowerCase();
        
        if (sortDirection === 'asc') {
          return strA.localeCompare(strB);
        } else {
          return strB.localeCompare(strA);
        }
      }
    });
    
    // Update the sorted records
    setSortedRecords(recordsCopy);
  }, [records, sortField, sortDirection]);
  
  // Add custom scrollbar styles on client side only
  useEffect(() => {
    const styleElement = document.createElement('style');
    styleElement.textContent = `
      .custom-scrollbar {
        -webkit-overflow-scrolling: touch;
        scrollbar-width: thin;
        scrollbar-color: #cbd5e0 #f7fafc;
      }
      .custom-scrollbar::-webkit-scrollbar {
        width: 8px;
        height: 8px;
      }
      .custom-scrollbar::-webkit-scrollbar-track {
        background: #f7fafc;
      }
      .custom-scrollbar::-webkit-scrollbar-thumb {
        background-color: #cbd5e0;
        border-radius: 4px;
      }
      .custom-scrollbar::-webkit-scrollbar-thumb:hover {
        background-color: #a0aec0;
      }
    `;
    
    // Add the style element to the document head
    document.head.appendChild(styleElement);
    
    // Clean up function to remove the style element when component unmounts
    return () => {
      document.head.removeChild(styleElement);
    };
  }, []);

  // Get the fields from the current table configuration
  const tableFields = currentTable?.fields || [];
  
  // Define the preferred column order based on the current table config
  const preferredOrder = currentTable?.preferredColumnOrder || 
    tableFields
      .filter(field => !field.hidden)
      .map(field => field.fieldName);

  // Fetch records when database selection or query params change
  useEffect(() => {
    let isMounted = true;
    
    async function fetchRecords() {
      // Don't fetch if we're still initializing
      if (!currentHostId || !currentDatabaseName || !currentTableName) {
        console.log('Waiting for initialization...', { 
          currentHostId, 
          currentDatabaseName, 
          currentTableName 
        });
        return;
      }
      
      console.log('Starting to fetch records with:', {
        host: currentHostId,
        db: currentDatabaseName,
        table: currentTableName,
        queryParams
      });
      
      setLoading(true);
      setError(null);
      
      try {
        // Ensure database and table names are properly encoded for URL
        const encodedDbName = encodeURIComponent(currentDatabaseName);
        const encodedTableName = encodeURIComponent(currentTableName);
        
        // Build the base URL without query parameters
        const baseUrl = `/api/odata/${encodedDbName}/${encodedTableName}`;
        
        // Construct the URL with query parameters
        let urlWithParams = baseUrl;
        
        // Check if we have query parameters
        if (queryParams) {
          // Don't use URLSearchParams for OData queries to preserve exact syntax
          // Just append the raw query string
          const separator = queryParams.startsWith('?') ? '' : '?';
          urlWithParams = `${baseUrl}${separator}${queryParams}`;
          
          // Add timestamp for cache busting
          const appendChar = urlWithParams.includes('?') ? '&' : '?';
          urlWithParams = `${urlWithParams}${appendChar}_t=${new Date().getTime().toString()}`;
        } else {
          // Just add timestamp if no other parameters
          urlWithParams = `${baseUrl}?_t=${new Date().getTime().toString()}`;
        }
        
        console.log('Query params:', queryParams);
        console.log('Final URL with params:', urlWithParams);
        
        console.log('Fetching records from:', urlWithParams);
        
        const response = await fetch(urlWithParams, {
          headers: {
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0',
            'Cache-Control': 'no-cache, no-store, must-revalidate',
            'Pragma': 'no-cache',
            'Expires': '0'
          },
          cache: 'no-store'
        });
        
        if (!isMounted) return;
        
        if (!response.ok) {
          const errorText = await response.text();
          console.error('Error response:', errorText);
          throw new Error(`Error fetching records: ${response.status} - ${response.statusText}`);
        }
        
        const data = await response.json();
        
        if (!isMounted) return;
        
        console.log('Raw API response type:', typeof data);
        console.log('Raw API response keys:', data ? Object.keys(data) : 'no data');
        console.log('Raw API response:', data);
        
        // Handle OData response format
        let recordsData = [];
        
        // First check if data is directly an array
        if (Array.isArray(data)) {
          console.log('Data is an array with length:', data.length);
          recordsData = data;
        } 
        // Check if data has a 'data' property containing an array (common in some APIs)
        else if (data && data.data && Array.isArray(data.data)) {
          console.log('Found records in data.data array with length:', data.data.length);
          recordsData = data.data;
        }
        // Check if data has a 'value' property (standard OData format)
        else if (data && typeof data === 'object' && data.hasOwnProperty('value')) {
          console.log('Data has value property. Type:', typeof data.value);
          console.log('Value property is array:', Array.isArray(data.value));
          console.log('Value property length:', Array.isArray(data.value) ? data.value.length : 'not an array');
          
          if (Array.isArray(data.value)) {
            recordsData = data.value;
          }
        }
        // If none of the above, check all properties for arrays
        else if (data && typeof data === 'object') {
          console.log('Attempting alternative data extraction methods');
          // Look for any array property that might contain records
          for (const key in data) {
            if (Array.isArray(data[key]) && data[key].length > 0) {
              console.log(`Found potential records in property: ${key} with ${data[key].length} items`);
              recordsData = data[key];
              break;
            }
          }
        }
        
        console.log('Final processed records data:', {
          isArray: Array.isArray(recordsData),
          recordCount: recordsData.length,
          firstRecord: recordsData.length > 0 ? 'found' : 'no records'
        });
        
        // Additional logging for debugging
        if (recordsData.length > 0) {
          console.log('Sample record keys:', Object.keys(recordsData[0]));
        }
        
        setRecords(recordsData);
      } catch (err: any) {
        console.error('Error in fetchRecords:', err);
        if (isMounted) {
          setError(err.message || 'Failed to fetch records');
        }
      } finally {
        if (isMounted) {
          setLoading(false);
        }
      }
    }

    fetchRecords();
    
    return () => {
      isMounted = false;
    };
  }, [currentHostId, currentDatabaseName, currentTableName, queryParams, refreshTrigger]);

  // Get hidden fields from the table configuration
  const hiddenFields = currentTable?.hiddenFields || [];
  
  // Extract column names from records and table configuration for display
  const tableColumns = (() => {
    // If we have a displayShort configuration, use that as the primary source of truth
    if (currentTable?.displayShort && currentTable.displayShort.length > 0) {
      console.log('Using displayShort configuration for columns');
      return currentTable.displayShort;
    }
    
    // Otherwise fall back to the original logic
    console.log('No displayShort configuration found, using fallback logic');
    
    // Get fields defined in the table configuration
    const tableFields = currentTable?.fields || [];
    const configFields = tableFields
      .filter(field => !field.hidden && !hiddenFields.includes(field.fieldName))
      .map(field => field.fieldName);
    
    // Define the preferred column order based on the current table config
    const preferredOrder = currentTable?.preferredColumnOrder || 
      tableFields
        .filter(field => !field.hidden && !hiddenFields.includes(field.fieldName))
        .map(field => field.fieldName);
    
    // If we have records, also include any fields from the records that aren't in the config
    if (records && records.length > 0) {
      // Get all unique keys from all records
      const allKeys = new Set<string>();
      records.forEach(record => {
        if (record && typeof record === 'object') {
          Object.keys(record).forEach(key => {
            // Filter out OData metadata fields and hidden fields
            if (!key.startsWith('@odata') && !hiddenFields.includes(key)) {
              allKeys.add(key);
            }
          });
        }
      });
      
      // Convert to array and sort based on preferred order
      const columns = Array.from(allKeys);
      return [
        // First include preferred columns in order
        ...preferredOrder.filter((col: string) => columns.includes(col)),
        // Then include any remaining columns from the config
        ...configFields.filter((col: string) => 
          !preferredOrder.includes(col) && columns.includes(col)
        ),
        // Finally include any columns from the data that aren't in the config or hidden
        ...columns.filter((col: string) => 
          !preferredOrder.includes(col) && 
          !configFields.includes(col) && 
          !hiddenFields.includes(col)
        ).sort()
      ];
    }
    
    // If no records, just use the config fields
    return configFields;
  })();

  // Get visible columns based on mobile/desktop view
  const getVisibleColumns = (): string[] => {
    if (isMobile) {
      return mobileFields.filter(field => 
        tableColumns.some(col => col === field)
      );
    }
    return tableColumns;
  };
  
  const visibleColumns = getVisibleColumns();
  
  // Format field name for display (e.g., 'firstName' -> 'First Name')
  const formatFieldName = (field: string): string => {
    // Mobile-specific field name formatting
    if (isMobile && mobileLabels[field]) {
      return mobileLabels[field];
    }
    
    // Check if we have a display name in the table configuration
    const fieldConfig = currentTable?.fields?.find(f => f.fieldName === field);
    if (fieldConfig?.displayName) {
      return fieldConfig.displayName;
    }
    
    // Default formatting for both mobile and desktop
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, (str: string) => str.toUpperCase())
      .trim();
  };

  // Format cell value for display based on field type
  const formatCellValue = (field: string, value: any): string => {
    if (value === null || value === undefined) return '-';
    
    // Handle Message field special formatting
    if (field === 'Message' && typeof value === 'string') {
      // Replace newline characters with ' - ' and trim any extra whitespace
      const formattedValue = value.replace(/[\r\n]+/g, ' - ').trim();
      // Truncate to 15 characters with ellipsis if needed
      return formattedValue.length > 15 ? `${formattedValue.substring(0, 15)}...` : formattedValue;
    }
    
    // Get field configuration if available
    const fieldConfig = currentTable?.fields?.find(f => f.fieldName === field);
    
    if (fieldConfig) {
      // Format based on field type
      switch (fieldConfig.type) {
        case 'date':
          try {
            const date = new Date(value);
            return date.toLocaleDateString();
          } catch (e) {
            return String(value);
          }
        case 'boolean':
          return value ? 'Yes' : 'No';
        case 'number':
          // Check if this is a record ID field
          if (field === 'z_recid' || field.toLowerCase().includes('id')) {
            return String(value); // Display IDs as simple numbers without formatting
          }
          return typeof value === 'number' ? value.toLocaleString() : String(value);
        default:
          if (typeof value === 'object') return JSON.stringify(value);
          return String(value);
      }
    }
    
    // Default formatting if no field config
    if (typeof value === 'object') return JSON.stringify(value);
    return String(value);
  };

  // Handle query input change
  const handleQueryInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setQueryInput(e.target.value);
  };
  
  // Handle search field change
  const handleSearchFieldChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchField(e.target.value);
  };
  
  // Handle search operation change
  const handleSearchOperationChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchOperation(e.target.value);
  };
  
  // Helper function to check if a string is in US date format (MM/DD/YYYY)
  const isUSDateFormat = (value: string): boolean => {
    // Check for MM/DD/YYYY format
    const usDateRegex = /^(0?[1-9]|1[0-2])\/(0?[1-9]|[12][0-9]|3[01])\/(\d{4})$/;
    return usDateRegex.test(value);
  };
  
  // Helper function to check for advanced date patterns
  const parseAdvancedDateQuery = (value: string): { type: string; dates: string[] } | null => {
    // Check for greater than pattern: >MM/DD/YYYY
    const greaterThanMatch = value.match(/^>\s*(\d{1,2}\/\d{1,2}\/\d{4})$/);
    if (greaterThanMatch && isUSDateFormat(greaterThanMatch[1])) {
      return { type: 'gt', dates: [greaterThanMatch[1]] };
    }
    
    // Check for less than pattern: <MM/DD/YYYY
    const lessThanMatch = value.match(/^<\s*(\d{1,2}\/\d{1,2}\/\d{4})$/);
    if (lessThanMatch && isUSDateFormat(lessThanMatch[1])) {
      return { type: 'lt', dates: [lessThanMatch[1]] };
    }
    
    // Check for date range pattern: from MM/DD/YYYY to MM/DD/YYYY
    const rangeMatch = value.match(/^from\s+(\d{1,2}\/\d{1,2}\/\d{4})\s+to\s+(\d{1,2}\/\d{1,2}\/\d{4})$/);
    if (rangeMatch && isUSDateFormat(rangeMatch[1]) && isUSDateFormat(rangeMatch[2])) {
      return { type: 'range', dates: [rangeMatch[1], rangeMatch[2]] };
    }
    
    // Not an advanced date query
    return null;
  };

  // Helper function to convert US date format to FileMaker OData API format
  // Add one day to account for how FileMaker OData API handles date conversions
  // Even though StartDate is a simple FileMaker date field (not timestamp),
  // the OData API's date handling requires this adjustment for correct results
  const convertUSDateToODataFormat = (usDate: string): string => {
    const parts = usDate.split('/');
    const month = parseInt(parts[0], 10);
    const day = parseInt(parts[1], 10);
    const year = parseInt(parts[2], 10);
    
    // Create a date object and add one day to account for timezone differences
    const date = new Date(year, month - 1, day);
    date.setDate(date.getDate() + 1); // Add one day
    
    // Format as YYYY-MM-DD without quotes for FileMaker OData API
    const adjustedYear = date.getFullYear();
    const adjustedMonth = (date.getMonth() + 1).toString().padStart(2, '0');
    const adjustedDay = date.getDate().toString().padStart(2, '0');
    
    return `${adjustedYear}-${adjustedMonth}-${adjustedDay}`;
  };

  // Handle form submission for query parameters
  const handleSubmitQuery = (e: React.FormEvent) => {
    e.preventDefault();
    
    // If we have a search field and query input, build an OData filter
    if (searchField && queryInput.trim()) {
      let filterExpression = '';
      const searchValue = queryInput.trim();
      
      // Special handling for date fields
      if (searchField === 'StartDate') {
        // Check for advanced date patterns first
        const advancedDateQuery = parseAdvancedDateQuery(searchValue);
        
        if (advancedDateQuery) {
          // Handle advanced date queries
          switch (advancedDateQuery.type) {
            case 'gt':
              const gtDate = convertUSDateToODataFormat(advancedDateQuery.dates[0]);
              filterExpression = `$filter=${searchField} gt ${gtDate}`;
              break;
            case 'lt':
              const ltDate = convertUSDateToODataFormat(advancedDateQuery.dates[0]);
              filterExpression = `$filter=${searchField} lt ${ltDate}`;
              break;
            case 'range':
              const fromDate = convertUSDateToODataFormat(advancedDateQuery.dates[0]);
              const toDate = convertUSDateToODataFormat(advancedDateQuery.dates[1]);
              filterExpression = `$filter=${searchField} ge ${fromDate} and ${searchField} le ${toDate}`;
              break;
          }
        } else if (isUSDateFormat(searchValue)) {
          // Standard date equality
          const odataDate = convertUSDateToODataFormat(searchValue);
          // FileMaker OData API expects dates without quotes in filter expressions
          filterExpression = `$filter=${searchField} eq ${odataDate}`;
        } else {
          // Invalid date format, use as-is (will likely return no results)
          filterExpression = `$filter=${searchField} ${searchOperation} '${searchValue}'`;
        }
      } else {
        // For non-date fields, use the standard filter expression
        // Format the value based on if it looks like a number or not
        const isNumeric = !isNaN(Number(searchValue)) && !isNaN(parseFloat(searchValue));
        const formattedValue = isNumeric ? searchValue : `'${searchValue}'`;
        
        // Build the filter expression based on the search operation
        // Use tolower() for case-insensitive searches
        if (searchOperation === 'contains') {
          filterExpression = `$filter=contains(tolower(${searchField}), tolower(${formattedValue}))`;
        } else if (searchOperation === 'equals') {
          filterExpression = `$filter=tolower(${searchField}) eq tolower(${formattedValue})`;
        } else {
          filterExpression = `$filter=${searchField} ${searchOperation} ${formattedValue}`;
        }
      }
      
      console.log('Generated filter expression:', filterExpression);
      setQueryParams(filterExpression);
      return;
    }
    
    // Fallback to the original query processing if no specific field/operation is used
    let processedInput = queryInput.trim();
    
    // Skip processing if it already starts with ? or $ (already formatted)
    if (!processedInput.startsWith('?') && !processedInput.startsWith('$') && !processedInput.includes('=')) {
      // Check for common incorrect syntax patterns and fix them
      // Example: "Name contains 'Dave'" should be "contains(Name, 'Dave')"
      const containsMatch = processedInput.match(/([\w]+)\s+contains\s+('[^']+'|"[^"]+")/i);
      if (containsMatch) {
        const [_, fieldName, value] = containsMatch;
        processedInput = `contains(${fieldName}, ${value})`;
        console.log('Fixed contains syntax:', processedInput);
      }
      
      // Check for OData function patterns like contains(), startswith(), etc.
      const odataFunctions = ['contains', 'startswith', 'endswith', 'indexof', 'substring', 'length', 'tolower', 'toupper', 'trim'];
      const hasFunctionPattern = odataFunctions.some(fn => {
        // Use a safe way to create the regex
        try {
          const pattern = new RegExp(fn + '\\s*\\(', 'i');
          return pattern.test(processedInput);
        } catch (e) {
          console.error('Invalid regex pattern for function:', fn, e);
          return false;
        }
      });
      
      // Check for OData comparison operators
      const hasComparisonOperator = (
        processedInput.includes(' eq ') || 
        processedInput.includes(' ne ') || 
        processedInput.includes(' gt ') || 
        processedInput.includes(' lt ') || 
        processedInput.includes(' ge ') || 
        processedInput.includes(' le ')
      );
      
      // If it looks like an OData expression, add the $filter= prefix
      if (hasComparisonOperator || hasFunctionPattern) {
        console.log('Detected OData filter expression, adding $filter= prefix');
        processedInput = `$filter=${processedInput}`;
      }
    }
    
    console.log('Submitting query:', processedInput);
    setQueryParams(processedInput);
  };

  // Handle refresh button click
  const handleRefresh = () => {
    setQueryInput('');
    setQueryParams('');
    setSortField('');
    setSortDirection('asc');
    // Re-trigger the effect by creating a new timestamp
    setRefreshTrigger(Date.now());
  }

  // Show loading state while initializing or loading data
  if (!currentHostId || !currentDatabaseName || !currentTableName) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex flex-col items-center justify-center h-64 space-y-4">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
              <p className="text-gray-600">Initializing database connection...</p>
              <div className="text-sm text-gray-500 bg-gray-50 p-2 rounded">
                <p>Host: {currentHostId || 'Loading...'}</p>
                <p>Database: {currentDatabaseName || 'Loading...'}</p>
                <p>Table: {currentTableName || 'Loading...'}</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex flex-col items-center justify-center h-64 space-y-4">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
              <p className="text-gray-600">Loading records...</p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                Record List
              </h1>
              <div className="flex gap-2">
                <Link
                  href="/update"
                  className="px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 text-sm"
                >
                  Add New
                </Link>
              </div>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4">
            <form onSubmit={handleSubmitQuery} className="flex flex-col gap-2 mb-6">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label htmlFor="searchField" className="block text-sm font-medium text-gray-700 mb-1">
                    Field
                  </label>
                  <select
                    id="searchField"
                    className="w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    value={searchField}
                    onChange={(e) => setSearchField(e.target.value)}
                  >
                    <option value="Name">Name</option>
                    <option value="Message">Message</option>
                    <option value="pl_Status">Status</option>
                    <option value="pl_Cat">Category</option>
                    <option value="StartDate">StartDate</option>
                  </select>
                </div>
                
                <div>
                  <label htmlFor="queryInput" className="block text-sm font-medium text-gray-700 mb-1">
                    Search Term
                  </label>
                  <input
                    type="text"
                    id="queryInput"
                    className="w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    value={queryInput}
                    onChange={handleQueryInputChange}
                    placeholder={searchField === 'StartDate' ? "Enter date (e.g., 6/7/2025, >6/7/2025)" : "Enter search term (e.g., Dave)"}
                  />
                  {searchField === 'StartDate' && (
                    <p className="mt-1 text-xs text-gray-500">
                      Formats: MM/DD/YYYY, &gt;MM/DD/YYYY, &lt;MM/DD/YYYY, or from MM/DD/YYYY to MM/DD/YYYY
                    </p>
                  )}
                </div>
                
                <div>
                  <fieldset className="mt-2">
                    <legend className="block text-sm font-medium text-gray-700 mb-1">Search Type</legend>
                    <div className="flex items-center space-x-6">
                      <div className="flex items-center">
                        <input
                          id="contains"
                          name="searchOperation"
                          type="radio"
                          value="contains"
                          checked={searchOperation === 'contains'}
                          onChange={handleSearchOperationChange}
                          className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <label htmlFor="contains" className="ml-2 block text-sm text-gray-700">
                          Contains
                        </label>
                      </div>
                      <div className="flex items-center">
                        <input
                          id="equals"
                          name="searchOperation"
                          type="radio"
                          value="equals"
                          checked={searchOperation === 'equals'}
                          onChange={handleSearchOperationChange}
                          className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <label htmlFor="equals" className="ml-2 block text-sm text-gray-700">
                          Equals
                        </label>
                      </div>
                    </div>
                  </fieldset>
                </div>
              </div>
              
              <div className="flex justify-end mt-2">
                <button
                  type="submit"
                  className="w-full sm:w-auto px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50"
                  disabled={loading}
                >
                  {loading ? 'Loading...' : 'Search'}
                </button>
                <button
                  type="button"
                  onClick={handleRefresh}
                  className="w-full sm:w-auto ml-2 px-4 py-2 bg-gray-200 text-gray-700 rounded hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-400 focus:ring-opacity-50"
                  disabled={loading}
                >
                  {loading ? 'Loading...' : 'Refresh'}
                </button>
              </div>
            </form>
            
            {error && (
              <div className="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {!loading && !error && records.length === 0 && (
              <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-yellow-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M8.485 2.495c.673-1.167 2.357-1.167 3.03 0l6.28 10.875c.673 1.167-.17 2.625-1.516 2.625H3.72c-1.347 0-2.189-1.458-1.515-2.625L8.485 2.495zM10 5a.75.75 0 01.75.75v3.5a.75.75 0 01-1.5 0v-3.5A.75.75 0 0110 5zm0 9a1 1 0 100-2 1 1 0 000 2z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-yellow-700">No records found.</p>
                  </div>
                </div>
              </div>
            )}

            {!loading && !error && records.length > 0 && (
              <div className="relative">
                <div className="overflow-x-auto w-full custom-scrollbar -mx-2 sm:mx-0">
                  <div className="min-w-full w-full px-2 sm:px-0">
                    <table className="min-w-full divide-y divide-gray-200">
                    <thead className="hidden sm:table-header-group">
                      <tr>
                        <th className="sticky left-0 z-10 px-3 py-3.5 text-left text-sm font-semibold text-gray-900 bg-gray-100 min-w-[120px]">
                          Actions
                        </th>
                        {visibleColumns.map((col: string) => {
                          // Check if this column should be sortable
                          const isSortable = ['Name', 'StartDate', 'pl_Status', 'pl_Cat'].includes(col);
                          const isActive = sortField === col;
                          
                          return (
                            <th
                              key={col}
                              className={`px-3 py-3.5 text-left text-sm font-semibold text-gray-900 bg-gray-100 ${isSortable ? 'cursor-pointer hover:bg-gray-200' : ''}`}
                              onClick={isSortable ? () => handleSort(col) : undefined}
                            >
                              <div className="flex items-center">
                                <span>{formatFieldName(col)}</span>
                                {isActive && (
                                  <span className="ml-1">
                                    {sortDirection === 'asc' ? '↑' : '↓'}
                                  </span>
                                )}
                              </div>
                            </th>
                          );
                        })}
                      </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-200 bg-white">
                      {sortedRecords.map((record: Record, index: number) => {
                        const recordId = getRecordId(record);
                        const clientId = getClientId(record);
                        
                        return (
                          <React.Fragment key={recordId || index}>
                            {/* Mobile Action Buttons Row */}
                            {isMobile && (
                              <tr className={`${index % 2 === 0 ? 'bg-white' : 'bg-gray-50'} border-b border-gray-200`}>
                                <td colSpan={visibleColumns.length + 1} className="px-3 py-2">
                                  <div className="flex flex-wrap gap-1.5">
                                    <Link
                                      href={`/task?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                      className="px-3 py-1.5 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-sm font-medium whitespace-nowrap shadow-sm"
                                    >
                                      View
                                    </Link>
                                    <Link
                                      href={`/task-update?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                      className="px-3 py-1.5 bg-blue-100 text-blue-700 rounded hover:bg-blue-200 text-sm font-medium whitespace-nowrap shadow-sm"
                                    >
                                      Edit
                                    </Link>
                                    <button
                                      className="px-3 py-1.5 bg-green-100 text-green-700 rounded hover:bg-green-200 text-sm font-medium whitespace-nowrap shadow-sm"
                                      onClick={() => clientId && (window.location.href = `/client?clid=${clientId}`)}
                                    >
                                      Client
                                    </button>
                                    <button
                                      className="px-3 py-1.5 bg-purple-100 text-purple-700 rounded hover:bg-purple-200 text-sm font-medium whitespace-nowrap shadow-sm"
                                      onClick={() => {
                                        if (!clientId) return;
                                        const taskName = record['Name'] || record['Task_Name'] || record['TaskName'] || record['Title'] || '';
                                        const clientIdFromNameField = record['Name_id'] !== undefined && record['Name_id'] !== null 
                                          ? String(record['Name_id']) 
                                          : '';
                                        sessionStorage.setItem('cl-name', taskName);
                                        sessionStorage.setItem('tsk-id', recordId);
                                        sessionStorage.setItem('clid', clientIdFromNameField);
                                        window.location.href = `/tasks?clid=${clientId}`;
                                      }}
                                    >
                                      Tasks
                                    </button>
                                  </div>
                                </td>
                              </tr>
                            )}
                            
                            {/* Data Row */}
                            <tr className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                              {/* Action Buttons - Desktop Only */}
                              {!isMobile && (
                                <td className="px-3 py-4 text-gray-800 w-[260px] sticky left-0 z-10 bg-white">
                                  <div className="flex flex-row flex-nowrap gap-1 md:gap-0.5">
                                    <Link
                                      href={`/task?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                      className="px-2 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-sm whitespace-nowrap"
                                    >
                                      View
                                    </Link>
                                    <Link
                                      href={`/task-update?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                      className="px-2 py-1 bg-blue-100 text-blue-700 rounded hover:bg-blue-200 text-sm whitespace-nowrap"
                                    >
                                      Edit
                                    </Link>
                                    <button
                                      className="px-2 py-1 bg-green-100 text-green-700 rounded hover:bg-green-200 text-sm whitespace-nowrap"
                                      onClick={() => clientId && (window.location.href = `/client?clid=${clientId}`)}
                                    >
                                      Client
                                    </button>
                                    <button
                                      className="px-2 py-1 bg-purple-100 text-purple-700 rounded hover:bg-purple-200 text-sm whitespace-nowrap"
                                      onClick={() => {
                                        if (!clientId) return;
                                        const taskName = record['Name'] || record['Task_Name'] || record['TaskName'] || record['Title'] || '';
                                        const clientIdFromNameField = record['Name_id'] !== undefined && record['Name_id'] !== null 
                                          ? String(record['Name_id']) 
                                          : '';
                                        sessionStorage.setItem('cl-name', taskName);
                                        sessionStorage.setItem('tsk-id', recordId);
                                        sessionStorage.setItem('clid', clientIdFromNameField);
                                        window.location.href = `/tasks?clid=${clientId}`;
                                      }}
                                    >
                                      Tasks
                                    </button>
                                  </div>
                                </td>
                              )}
                              
                              {/* Data Cells */}
                              {visibleColumns.map((col: string) => (
                                <td 
                                  key={col}
                                  className={`whitespace-nowrap px-3 py-3 text-gray-800 text-sm sm:text-base ${col === 'Message' ? 'max-w-[180px] truncate text-[20px] sm:text-base' : ''} border-b border-gray-100`}
                                >
                                  {formatCellValue(col, record[col])}
                                </td>
                              ))}
                            </tr>
                          </React.Fragment>
                        );
                      })}
                    </tbody>
                    </table>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
