'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import DatabaseSelector from '@/components/DatabaseSelector';
import NavBar from '@/components/NavBar';
import ScriptDemo from '@/components/ScriptDemo';

export default function Home() {
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName 
  } = useDatabaseContext();
  
  const [loginInfo, setLoginInfo] = useState<{name: string | null; id: string | null}>({name: null, id: null});
  
  useEffect(() => {
    // This code runs only on the client side
    if (typeof window !== 'undefined') {
      setLoginInfo({
        name: sessionStorage.getItem('login-name'),
        id: sessionStorage.getItem('login-id')
      });
    }
  }, []);
  return (
    <div className="min-h-screen bg-gray-50">
      <NavBar />
      {loginInfo.name && (
        <div className="bg-blue-50 border-b border-blue-200">
          <div className="max-w-7xl mx-auto px-4 py-3 sm:px-6 lg:px-8">
            <div className="flex items-center justify-between flex-wrap">
              <div className="w-0 flex-1 flex items-center">
                <span className="flex p-2 rounded-lg bg-blue-600">
                  <svg className="h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M10 9a3 3 0 100-6 3 3 0 000 6zm-7 9a7 7 0 1114 0H3z" clipRule="evenodd" />
                  </svg>
                </span>
                <p className="ml-3 font-medium text-blue-800 truncate">
                  <span className="hidden md:inline">Logged in as </span>
                  <span className="font-semibold">{loginInfo.name}</span>
                  {loginInfo.id && (
                    <span className="text-blue-700 text-sm ml-2">(ID: {loginInfo.id})</span>
                  )}
                </p>
              </div>
            </div>
          </div>
        </div>
      )}
      <div className="p-6">
        <div className="max-w-4xl mx-auto">
          <div className="bg-white rounded-lg shadow-lg overflow-hidden">
            <div className="px-6 py-4 border-b border-gray-200">
              <h1 className="text-2xl font-semibold text-gray-900">FileMaker OData API Dashboard</h1>
            </div>
            
            <div className="p-6">
              <div className="mb-8">
                <h2 className="text-xl font-medium text-gray-800 mb-4">Navigation</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <Link 
                    href={`/list?host=${currentHostId}&database=${currentDatabaseName}&table=${currentTableName}`} 
                    className="flex flex-col items-center justify-center p-6 bg-blue-50 rounded-lg border border-blue-100 hover:bg-blue-100 transition-colors"
                  >
                    <div className="w-12 h-12 bg-blue-500 rounded-full flex items-center justify-center mb-3">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 10h16M4 14h16M4 18h16" />
                      </svg>
                    </div>
                    <span className="text-lg font-medium text-blue-700">Records List</span>
                    <p className="text-sm text-blue-600 text-center mt-2">View and manage all records</p>
                  </Link>
                  
                  <Link 
                    href={`/single?clid=2&host=${currentHostId}&database=${currentDatabaseName}&table=${currentTableName}`} 
                    className="flex flex-col items-center justify-center p-6 bg-green-50 rounded-lg border border-green-100 hover:bg-green-100 transition-colors"
                  >
                    <div className="w-12 h-12 bg-green-500 rounded-full flex items-center justify-center mb-3">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                      </svg>
                    </div>
                    <span className="text-lg font-medium text-green-700">View Record</span>
                    <p className="text-sm text-green-600 text-center mt-2">View details of record #2</p>
                  </Link>
                  
                  <Link 
                    href={`/update?id=2&host=${currentHostId}&database=${currentDatabaseName}&table=${currentTableName}`} 
                    className="flex flex-col items-center justify-center p-6 bg-amber-50 rounded-lg border border-amber-100 hover:bg-amber-100 transition-colors"
                  >
                    <div className="w-12 h-12 bg-amber-500 rounded-full flex items-center justify-center mb-3">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                      </svg>
                    </div>
                    <span className="text-lg font-medium text-amber-700">Update Record</span>
                    <p className="text-sm text-amber-600 text-center mt-2">Edit and update record #2</p>
                  </Link>
                </div>
              </div>
              
              <div className="mb-8">
                <h2 className="text-xl font-medium text-gray-800 mb-4">Additional Resources</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Link 
                    href="/ztest" 
                    className="flex items-center p-4 bg-purple-50 rounded-lg border border-purple-100 hover:bg-purple-100 transition-colors"
                  >
                    <div className="w-10 h-10 bg-purple-500 rounded-full flex items-center justify-center mr-4">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19.428 15.428a2 2 0 00-1.022-.547l-2.387-.477a6 6 0 00-3.86.517l-.318.158a6 6 0 01-3.86.517L6.05 15.21a2 2 0 00-1.806.547M8 4h8l-1 1v5.172a2 2 0 00.586 1.414l5 5c1.26 1.26.367 3.414-1.415 3.414H4.828c-1.782 0-2.674-2.154-1.414-3.414l5-5A2 2 0 009 10.172V5L8 4z" />
                      </svg>
                    </div>
                    <div>
                      <span className="text-lg font-medium text-purple-700">Test Page</span>
                      <p className="text-sm text-purple-600">Original test interface with all features</p>
                    </div>
                  </Link>
                  
                  <a 
                    href="https://docs.microsoft.com/en-us/odata/overview" 
                    target="_blank"
                    rel="noopener noreferrer"
                    className="flex items-center p-4 bg-gray-50 rounded-lg border border-gray-200 hover:bg-gray-100 transition-colors"
                  >
                    <div className="w-10 h-10 bg-gray-500 rounded-full flex items-center justify-center mr-4">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253" />
                      </svg>
                    </div>
                    <div>
                      <span className="text-lg font-medium text-gray-700">OData Documentation</span>
                      <p className="text-sm text-gray-600">Learn more about OData API standards</p>
                    </div>
                  </a>
                </div>
              </div>
              
              <div className="mt-12">
                <ScriptDemo 
                  hostId={currentHostId}
                  databaseName={currentDatabaseName}
                  tableName={currentTableName}
                />
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
