'use client';

import { useState, useEffect, useMemo, useCallback, ReactNode } from 'react';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import NavBar from '@/components/NavBar';
import { 
  clientsMobileFields, 
  clientsMobileLabels, 
  clientsDisplayFields 
} from '@/lib/config/field-names';

// Define field type for table configuration
interface TableField {
  fieldName: string;
  hidden?: boolean;
  [key: string]: any;
}

interface ClientRecord {
  [key: string]: any;
  id?: string;
  ID?: string;
  Id?: string;
  Name?: string;
  Email?: string;
  Phone?: string;
  Status?: string;
  CreatedDate?: string;
  ModifiedDate?: string;
}

export default function ListPage() {
  // Database context
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentTableName
  } = useDatabaseContext();

  // State management
  const [records, setRecords] = useState<ClientRecord[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);
  const [queryParams, setQueryParams] = useState<string>('$filter=Web eq 1');
  const [queryInput, setQueryInput] = useState<string>('');
  const [searchField, setSearchField] = useState<string>('Name');
  const [searchOperation, setSearchOperation] = useState<string>('contains');
  const [sortField, setSortField] = useState<string>('Name');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('asc');
  const [isMobile, setIsMobile] = useState<boolean>(false);

  // Mobile viewport detection and cleanup
  useEffect(() => {
    const checkIfMobile = () => {
      setIsMobile(window.innerWidth <= 768);
    };
    
    // Initial check
    checkIfMobile();
    
    // Add event listener for window resize
    window.addEventListener('resize', checkIfMobile);
    
    // Cleanup
    return () => {
      window.removeEventListener('resize', checkIfMobile);
    };
  }, []);

  // Use mobile field configuration from field-names.ts
  const mobileFields = clientsMobileFields;
  const mobileLabels = clientsMobileLabels;

  // Use display fields from field-names.ts
  const defaultColumns = useMemo(() => clientsDisplayFields as readonly string[], []);

  // Always set table to 'Clients' when component mounts
  useEffect(() => {
    console.log('Setting table to Clients. Current host/db:', { currentHostId, currentDatabaseName });
    setCurrentTableName('Clients');
  }, [setCurrentTableName, currentHostId, currentDatabaseName]);

  // Debug current context values
  useEffect(() => {
    console.log('Current database context values:', {
      host: currentHostId,
      database: currentDatabaseName,
      table: currentTableName,
      tableConfig: currentTable
    });
  }, [currentHostId, currentDatabaseName, currentTableName, currentTable]);

  // Get fields and hidden fields from table configuration or use defaults
  const { tableFields, hiddenFields } = useMemo(() => {
    const fields = (currentTable?.fields || []).length > 0 
      ? currentTable!.fields 
      : defaultColumns.map(field => ({ fieldName: field }));
      
    const hidden = currentTable?.hiddenFields || [];
    
    return { tableFields: fields, hiddenFields: hidden };
  }, [currentTable, defaultColumns]);

  // Format field name for display (with mobile-specific formatting)
  const formatFieldName = useCallback((field: string): string => {
    if (isMobile && field in mobileLabels) {
      return mobileLabels[field];
    }
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, (str: string) => str.toUpperCase())
      .trim();
  }, [isMobile, mobileLabels]);

  // Format cell value for display based on field type
  const formatCellValue = useCallback((field: string, value: any): string => {
    if (value === null || value === undefined) return '-';
    
    // Handle date fields
    if ((field === 'CreatedDate' || field === 'ModifiedDate') && value) {
      try {
        const date = new Date(value);
        return date.toLocaleDateString();
      } catch (e) {
        console.error('Error formatting date:', e);
      }
    }
    
    // Handle other value types
    if (typeof value === 'boolean') return value ? 'Yes' : 'No';
    if (typeof value === 'number') return value.toLocaleString();
    if (typeof value === 'string') return value;
    
    // Default formatting if no field config
    if (typeof value === 'object') return JSON.stringify(value);
    return String(value);
  }, []);

  // Helper function to get client ID from record
  // Only uses Name_id field and throws an error if not available
  const getClientId = useCallback((record: RecordWithId | null | undefined): string => {
    // Handle null or undefined record
    if (!record || typeof record !== 'object') {
      const error = new Error('Invalid record provided to getClientId');
      console.error(error.message, record);
      throw error;
    }
    
    // Try to get Name_id (case-insensitive check)
    const nameIdKey = Object.keys(record).find(
      key => key === 'Name_id' || key === 'Name_ID' || key === 'name_id'
    );
    
    // If we found Name_id and it has a value, use it
    if (nameIdKey && record[nameIdKey] != null && record[nameIdKey] !== '') {
      return String(record[nameIdKey]);
    }
    
    // If we get here, no valid Name_id was found
    const error = new Error(`Name_id is required but not found in record. Available fields: ${Object.keys(record).join(', ')}`);
    console.error(error.message, record);
    throw error;
  }, []);

  // Get visible columns based on mobile/desktop view
  const getVisibleColumns = useCallback((): string[] => {
    if (isMobile) {
      return mobileFields.filter((field: string) => 
        tableFields.some((f: TableField) => f.fieldName === field)
      );
    }
    return tableFields
      .filter((field: TableField) => !field.hidden)
      .map((field: TableField) => field.fieldName);
  }, [isMobile, tableFields, mobileFields]);

  // Get the visible columns based on current view
  const visibleColumns = useMemo(() => getVisibleColumns(), [getVisibleColumns]);

  // Define table column type
  interface TableColumn {
    key: string;
    field: string;
    headerName: string;
    width: number;
    renderCell: (params: { row: Record<string, any> }) => ReactNode;
  }

  // Define table columns based on visible columns
  const tableColumns = useMemo<TableColumn[]>(() => {
    return visibleColumns.map((column: string) => ({
      key: column,
      field: column,
      headerName: formatFieldName(column),
      width: 150,
      renderCell: (params: { row: Record<string, any> }) => (
        <div className="truncate">
          {formatCellValue(column, params.row[column])}
        </div>
      ),
    }));
  }, [visibleColumns, formatFieldName, formatCellValue]);

  // Handle column sorting
  const handleSort = (field: string) => {
    if (sortField === field) {
      setSortDirection(sortDirection === 'asc' ? 'desc' : 'asc');
    } else {
      setSortField(field);
      setSortDirection('asc');
    }
  };
  

  // Fetch records when database selection or query params change
  useEffect(() => {
    let isMounted = true;
    
    async function fetchRecords() {
      // Don't fetch if we're still initializing
      if (!currentHostId || !currentDatabaseName || !currentTableName) {
        console.log('Waiting for initialization...', { 
          currentHostId, 
          currentDatabaseName, 
          currentTableName 
        });
        return;
      }
      
      console.log('Starting to fetch records with:', {
        host: currentHostId,
        db: currentDatabaseName,
        table: currentTableName,
        queryParams
      });
      
      setLoading(true);
      setError(null);
      
      try {
        // Ensure database and table names are properly encoded for URL
        const encodedDbName = encodeURIComponent(currentDatabaseName);
        const encodedTableName = encodeURIComponent(currentTableName);
        
        // Build the base URL without query parameters
        const baseUrl = `/api/odata/${encodedDbName}/${encodedTableName}`;
        
        // Handle query parameters
        let finalQueryString = queryParams.startsWith('?') ? queryParams.substring(1) : queryParams;
        
        // Add timestamp for cache busting
        const timestamp = `_t=${new Date().getTime().toString()}`;
        finalQueryString = finalQueryString ? `${finalQueryString}&${timestamp}` : timestamp;
        
        // For debugging, log the raw query string before sending
        console.log('Raw query string:', finalQueryString);
        
        // Construct the final URL with query parameters
        // Don't use URLSearchParams as it can interfere with OData syntax
        const urlWithParams = `${baseUrl}?${finalQueryString}`;
        
        console.log('Fetching records from:', urlWithParams);
        
        const response = await fetch(urlWithParams, {
          headers: {
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0',
            'Cache-Control': 'no-cache, no-store, must-revalidate',
            'Pragma': 'no-cache',
            'Expires': '0'
          },
          cache: 'no-store'
        });
        
        if (!isMounted) return;
        
        if (!response.ok) {
          const errorText = await response.text();
          console.error('Error response:', errorText);
          throw new Error(`Error fetching records: ${response.status} - ${response.statusText}`);
        }
        
        const data = await response.json();
        
        if (!isMounted) return;
        
        console.log('Raw API response type:', typeof data);
        console.log('Raw API response keys:', data ? Object.keys(data) : 'no data');
        console.log('Raw API response:', data);
        
        // Handle OData response format
        let recordsData = [];
        if (Array.isArray(data)) {
          console.log('Data is an array with length:', data.length);
          recordsData = data;
        } else if (data && typeof data === 'object') {
          // Handle OData response with value property
          console.log('Data is an object. Has value property:', data.hasOwnProperty('value'));
          if (data.hasOwnProperty('value')) {
            console.log('Value property type:', typeof data.value);
            console.log('Value property is array:', Array.isArray(data.value));
            console.log('Value property length:', Array.isArray(data.value) ? data.value.length : 'not an array');
          }
          recordsData = data.value || [];
        }
        
        console.log('Processed records data:', {
          isArray: Array.isArray(recordsData),
          recordCount: recordsData.length,
          firstRecord: recordsData[0] || 'no records'
        });
        
        // If we still have no records, try to extract data differently
        if (recordsData.length === 0 && data && typeof data === 'object') {
          console.log('Attempting alternative data extraction methods');
          // Look for any array property that might contain records
          for (const key in data) {
            if (Array.isArray(data[key]) && data[key].length > 0) {
              console.log(`Found potential records in property: ${key} with ${data[key].length} items`);
              recordsData = data[key];
              break;
            }
          }
        }
        
        setRecords(recordsData);
      } catch (err: any) {
        console.error('Error in fetchRecords:', err);
        if (isMounted) {
          setError(err.message || 'Failed to fetch records');
        }
      } finally {
        if (isMounted) {
          setLoading(false);
        }
      }
    }

    fetchRecords();
    
    return () => {
      isMounted = false;
    };
  }, [currentHostId, currentDatabaseName, currentTableName, queryParams]);

  // Debug: Log records when they change
  useEffect(() => {
    if (records.length > 0) {
      console.log('Records loaded. First record structure:', {
        keys: Object.keys(records[0]),
        firstRecord: records[0]
      });
    }
  }, [records]);

  // hiddenFields is now derived in the useMemo hook above
  
  // Extract column names from records and table configuration for display
  const tableColumnConfig = useMemo(() => {
    // If we have a displayShort configuration, use that as the primary source of truth
    if (currentTable?.displayShort && currentTable.displayShort.length > 0) {
      console.log('Using displayShort configuration for clients columns');
      return currentTable.displayShort;
    }
    
    // Otherwise fall back to the original logic
    console.log('No displayShort configuration found, using fallback logic');
    
    // Use the memoized tableFields from the outer scope with proper typing
    const configFields = tableFields
      .filter((field: TableField) => !field.hidden && !hiddenFields.includes(field.fieldName))
      .map((field: TableField) => field.fieldName);
    
    // Define the preferred column order based on the current table config
    const preferredOrder = currentTable?.preferredColumnOrder || 
      tableFields
        .filter((field: TableField) => !field.hidden && !hiddenFields.includes(field.fieldName))
        .map((field: TableField) => field.fieldName);
    
    // If we have records, also include any fields from the records that aren't in the config
    if (records && records.length > 0) {
      // Get all unique keys from all records
      const recordKeys = new Set<string>();
      records.forEach(record => {
        if (record && typeof record === 'object') {
          Object.keys(record).forEach(key => {
            // Filter out OData metadata fields and hidden fields
            if (!key.startsWith('@odata') && !hiddenFields.includes(key)) {
              recordKeys.add(key);
            }
          });
        }
      });
      
      // Convert to array and sort based on preferred order
      const columns = Array.from(recordKeys);
      return [
        // First include preferred columns in order
        ...preferredOrder.filter((col: string) => columns.includes(col)),
        // Then include any remaining columns from the config
        ...configFields.filter((col: string) => 
          !preferredOrder.includes(col) && columns.includes(col)
        ),
        // Finally include any columns from the data that aren't in the config or hidden
        ...columns.filter((col: string) => 
          !preferredOrder.includes(col) && 
          !configFields.includes(col) && 
          !hiddenFields.includes(col)
        ).sort()
      ];
    } else {
      // If no records, just use the config fields
      const allRecordKeys = records.length > 0 
        ? Object.keys(records[0])
        : [];
      const columns = Array.from(new Set([...allRecordKeys, ...configFields]));
      return [
        // First include preferred columns in order
        ...preferredOrder.filter((col: string) => columns.includes(col)),
        // Then include any remaining columns from the config
        ...configFields.filter((col: string) => 
          !preferredOrder.includes(col) && columns.includes(col)
        ),
        // Finally include any columns from the data that aren't in the config or hidden
        ...columns.filter((col: string) => 
          !preferredOrder.includes(col) && 
          !configFields.includes(col) && 
          !hiddenFields.includes(col)
        ).sort()
      ];
    }
  }, [currentTable, records, hiddenFields]);

  // Define a more specific type for the record parameter
  interface RecordWithId {
    [key: string]: any;
    '@odata.id'?: string;
    '@id'?: string;
    Name_id?: string | number;
  }

  // Extract record ID from @id field or other potential ID fields
  const getRecordId = (record: RecordWithId): string => {
    // Debug log the record to see available fields
    console.log('Getting ID for record:', record);
    
    // Prioritize Name_id field as specified
    if (record['Name_id'] !== undefined && record['Name_id'] !== null) {
      console.log('Found Name_id field:', record['Name_id']);
      return String(record['Name_id']);
    }
    
    // Check @odata.id field (standard OData field)
    if (record['@odata.id']) {
      console.log('Found @odata.id:', record['@odata.id']);
      const match = record['@odata.id'].match(/\((\d+)\)/);
      if (match && match[1]) {
        console.log('Extracted ID from @odata.id:', match[1]);
        return match[1];
      }
    }
    
    // Check @id field (some OData implementations use this)
    if (record['@id']) {
      console.log('Found @id:', record['@id']);
      const match = record['@id'].match(/\((\d+)\)/);
      if (match && match[1]) {
        console.log('Extracted ID from @id:', match[1]);
        return match[1];
      }
    }
    
    // Fallback to other ID fields if URL extraction fails
    const possibleIdFields = [
      // Common ID field names
      'ID', 'Id', 'id', 'z_recid', 'recid', '_id',
      // FileMaker specific
      'RecordID', 'record_id', 'Nid', 'nid', 'ctid',
      // Other common patterns
      'key', 'primaryKey', 'recordId', 'recordKey'
    ];
    
    // Try exact matches first
    for (const idField of possibleIdFields) {
      if (record[idField] !== undefined && record[idField] !== null) {
        console.log(`Found ID in field '${idField}':`, record[idField]);
        return String(record[idField]);
      }
    }
    
    // Log available fields for debugging
    console.log('Available fields in record:', Object.keys(record));
    
    // If no ID is found, try to find any field that might contain an ID
    for (const [key, value] of Object.entries(record)) {
      // Skip metadata fields and complex objects
      if (key.startsWith('@') || key.startsWith('_') || typeof value === 'object') {
        continue;
      }
      
      // If the field name suggests it might be an ID
      const lowerKey = key.toLowerCase();
      if ((lowerKey.includes('id') || lowerKey.includes('rec') || lowerKey.endsWith('_id')) && 
          value !== null && value !== undefined) {
        console.log(`Using fallback ID field '${key}':`, value);
        return String(value);
      }
    }
    
    console.warn('Could not determine record ID for record:', record);
    // Return a negative number based on the record's position in memory
    // This ensures a unique value even when no ID field is found
    return '-' + Math.abs(Object.entries(record).toString().length);
  };

  // Helper function to check if a string is in US date format (MM/DD/YYYY)
  const isUSDateFormat = (value: string): boolean => {
    // Check for MM/DD/YYYY format
    const usDateRegex = /^(0?[1-9]|1[0-2])\/(0?[1-9]|[12][0-9]|3[01])\/(\d{4})$/;
    return usDateRegex.test(value);
  };
  
  // Helper function to check for advanced date patterns
  const parseAdvancedDateQuery = (value: string): { type: string; dates: string[] } | null => {
    // Check for greater than pattern: >MM/DD/YYYY
    const greaterThanMatch = value.match(/^>\s*(\d{1,2}\/\d{1,2}\/\d{4})$/);
    if (greaterThanMatch && isUSDateFormat(greaterThanMatch[1])) {
      return { type: 'gt', dates: [greaterThanMatch[1]] };
    }
    
    // Check for less than pattern: <MM/DD/YYYY
    const lessThanMatch = value.match(/^<\s*(\d{1,2}\/\d{1,2}\/\d{4})$/);
    if (lessThanMatch && isUSDateFormat(lessThanMatch[1])) {
      return { type: 'lt', dates: [lessThanMatch[1]] };
    }
    
    // Check for date range pattern: from MM/DD/YYYY to MM/DD/YYYY
    const rangeMatch = value.match(/^from\s+(\d{1,2}\/\d{1,2}\/\d{4})\s+to\s+(\d{1,2}\/\d{1,2}\/\d{4})$/);
    if (rangeMatch && isUSDateFormat(rangeMatch[1]) && isUSDateFormat(rangeMatch[2])) {
      return { type: 'range', dates: [rangeMatch[1], rangeMatch[2]] };
    }
    
    // Not an advanced date query
    return null;
  };

  // Helper function to convert US date format to FileMaker OData API format
  // Add one day to account for how FileMaker OData API handles date conversions
  // Even though date fields are simple FileMaker date fields (not timestamps),
  // the OData API's date handling requires this adjustment for correct results
  const convertUSDateToODataFormat = (usDate: string): string => {
    const parts = usDate.split('/');
    const month = parseInt(parts[0], 10);
    const day = parseInt(parts[1], 10);
    const year = parseInt(parts[2], 10);
    
    // Create a date object and add one day to account for timezone differences
    const date = new Date(year, month - 1, day);
    date.setDate(date.getDate() + 1); // Add one day
    
    // Format as YYYY-MM-DD without quotes for FileMaker OData API
    const adjustedYear = date.getFullYear();
    const adjustedMonth = (date.getMonth() + 1).toString().padStart(2, '0');
    const adjustedDay = date.getDate().toString().padStart(2, '0');
    
    return `${adjustedYear}-${adjustedMonth}-${adjustedDay}`;
  };

  // Handle query input change without triggering data fetch
  const handleQueryInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setQueryInput(e.target.value);
  };
  
  // Search field change is handled inline with onChange={(e) => setSearchField(e.target.value)}
  
  // Handle search operation change
  const handleSearchOperationChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchOperation(e.target.value);
  };

  // Helper function to combine Web filter with other filters
  const combineWithWebFilter = (filter: string): string => {
    if (!filter) return '$filter=Web eq 1';
    
    // Remove any existing $filter= prefix
    let cleanFilter = filter.replace(/^\$filter=/, '');
    
    // If the filter already contains Web eq 1, return as is
    if (cleanFilter.includes('Web eq 1')) {
      return filter;
    }
    
    // Combine the new filter with Web eq 1
    return `$filter=(${cleanFilter}) and (Web eq 1)`;
  };

  // Handle form submission to update query params and trigger data fetch
  const handleSubmitQuery = (e: React.FormEvent) => {
    e.preventDefault();
    
    // If we have a search field and query input, build an OData filter
    if (searchField && queryInput.trim()) {
      let filterExpression = '';
      const searchValue = queryInput.trim();
      
      // Special handling for date fields
      if (searchField === 'CreatedDate' || searchField === 'ModifiedDate') {
        // Check for advanced date patterns first
        const advancedDateQuery = parseAdvancedDateQuery(searchValue);
        
        if (advancedDateQuery) {
          // Handle advanced date queries
          switch (advancedDateQuery.type) {
            case 'gt':
              const gtDate = convertUSDateToODataFormat(advancedDateQuery.dates[0]);
              filterExpression = `${searchField} gt ${gtDate}`;
              break;
            case 'lt':
              const ltDate = convertUSDateToODataFormat(advancedDateQuery.dates[0]);
              filterExpression = `${searchField} lt ${ltDate}`;
              break;
            case 'range':
              const fromDate = convertUSDateToODataFormat(advancedDateQuery.dates[0]);
              const toDate = convertUSDateToODataFormat(advancedDateQuery.dates[1]);
              filterExpression = `${searchField} ge ${fromDate} and ${searchField} le ${toDate}`;
              break;
          }
        } else if (isUSDateFormat(searchValue)) {
          // Standard date equality
          const odataDate = convertUSDateToODataFormat(searchValue);
          // FileMaker OData API expects dates without quotes in filter expressions
          filterExpression = `${searchField} eq ${odataDate}`;
        } else {
          // Invalid date format, use as-is (will likely return no results)
          filterExpression = `${searchField} ${searchOperation} '${searchValue}'`;
        }
      } else {
        // For non-date fields, use the standard filter expression
        // Format the value based on if it looks like a number or not
        const isNumeric = !isNaN(Number(searchValue)) && !isNaN(parseFloat(searchValue));
        const formattedValue = isNumeric ? searchValue : `'${searchValue.replace(/'/g, "''")}'`;
        
        // Build the filter expression based on the search operation
        // Use tolower() for case-insensitive searches
        if (searchOperation === 'contains') {
          filterExpression = `contains(tolower(${searchField}), tolower(${formattedValue}))`;
        } else if (searchOperation === 'equals') {
          filterExpression = `tolower(${searchField}) eq tolower(${formattedValue})`;
        } else {
          filterExpression = `${searchField} ${searchOperation} ${formattedValue}`;
        }
      }
      
      console.log('Generated filter expression:', filterExpression);
      // Combine with Web eq 1 filter
      const combinedFilter = combineWithWebFilter(filterExpression);
      console.log('Combined filter:', combinedFilter);
      setQueryParams(combinedFilter);
      return;
    }
    
    // Fallback to the original query processing if no specific field/operation is used
    let processedInput = queryInput.trim();
    
    // Skip processing if it already starts with ? or $ (already formatted)
    if (!processedInput.startsWith('?') && !processedInput.startsWith('$') && !processedInput.includes('=')) {
      // Check for common incorrect syntax patterns and fix them
      // Example: "Name contains 'Dave'" should be "contains(Name, 'Dave')"
      const containsMatch = processedInput.match(/([\w]+)\s+contains\s+('[^']+'|\"[^\"]+\")/i);
      if (containsMatch) {
        const [_, fieldName, value] = containsMatch;
        processedInput = `contains(${fieldName}, ${value})`;
        console.log('Fixed contains syntax:', processedInput);
      }
      
      // If it looks like an OData expression, add the $filter= prefix
      if (processedInput) {
        console.log('Adding $filter= prefix to query');
        processedInput = `$filter=${processedInput}`;
      }
    }
    
    console.log('Submitting query:', processedInput);
    // Combine with Web eq 1 filter
    const combinedFilter = combineWithWebFilter(processedInput);
    console.log('Combined filter:', combinedFilter);
    setQueryParams(combinedFilter);
  };

  // Handle refresh button click
  const handleRefresh = () => {
    // Reset to default filter with Web eq 1
    setQueryParams('$filter=Web eq 1');
    setQueryInput('');
    setSearchField('Name');
    setSearchOperation('contains');
  }
  
  // We don't want to update queryInput when queryParams changes
  // This allows the search term to remain in the input field after search

  // Show loading state while initializing or loading data
  if (!currentHostId || !currentDatabaseName || !currentTableName) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex flex-col items-center justify-center h-64 space-y-4">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
              <p className="text-gray-600">Initializing database connection...</p>
              <div className="text-sm text-gray-500 bg-gray-50 p-2 rounded">
                <p>Host: {currentHostId || 'Loading...'}</p>
                <p>Database: {currentDatabaseName || 'Loading...'}</p>
                <p>Table: {currentTableName || 'Loading...'}</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex flex-col items-center justify-center h-64 space-y-4">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
              <p className="text-gray-600">Loading records...</p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                Record List
              </h1>
              <div className="flex gap-2">
                <Link
                  href="/client-new"
                  className="px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 text-sm"
                >
                  Add New
                </Link>
              </div>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4">
            <form onSubmit={handleSubmitQuery} className="flex flex-col gap-4 mb-6">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label htmlFor="searchField" className="block text-sm font-medium text-gray-700 mb-1">
                    Search Field
                  </label>
                  <select
                    id="searchField"
                    className="w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    value={searchField}
                    onChange={(e) => setSearchField(e.target.value)}
                  >
                    <option value="Name">Name</option>
                    <option value="Email">Email</option>
                    <option value="Phone">Phone</option>
                    <option value="Company">Company</option>
                    <option value="Status">Status</option>
                    <option value="CreatedDate">Created Date</option>
                    <option value="ModifiedDate">Modified Date</option>
                  </select>
                </div>
                
                <div>
                  <label htmlFor="queryInput" className="block text-sm font-medium text-gray-700 mb-1">
                    Search Term
                  </label>
                  <input
                    type="text"
                    id="queryInput"
                    className="w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    value={queryInput}
                    onChange={handleQueryInputChange}
                    placeholder={searchField === 'CreatedDate' || searchField === 'ModifiedDate' ? "Enter date (e.g., 6/7/2025, >6/7/2025)" : "Enter search term (e.g., Dave)"}
                  />
                  {(searchField === 'CreatedDate' || searchField === 'ModifiedDate') && (
                    <p className="mt-1 text-xs text-gray-500">
                      Formats: MM/DD/YYYY, &gt;MM/DD/YYYY, &lt;MM/DD/YYYY, or from MM/DD/YYYY to MM/DD/YYYY
                    </p>
                  )}
                </div>
                
                <div>
                  <fieldset className="mt-2">
                    <legend className="block text-sm font-medium text-gray-700 mb-1">Search Type</legend>
                    <div className="flex items-center space-x-6">
                      <div className="flex items-center">
                        <input
                          id="contains"
                          name="searchOperation"
                          type="radio"
                          value="contains"
                          checked={searchOperation === 'contains'}
                          onChange={handleSearchOperationChange}
                          className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <label htmlFor="contains" className="ml-2 block text-sm text-gray-700">
                          Contains
                        </label>
                      </div>
                      <div className="flex items-center">
                        <input
                          id="equals"
                          name="searchOperation"
                          type="radio"
                          value="equals"
                          checked={searchOperation === 'equals'}
                          onChange={handleSearchOperationChange}
                          className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <label htmlFor="equals" className="ml-2 block text-sm text-gray-700">
                          Equals
                        </label>
                      </div>
                    </div>
                  </fieldset>
                </div>
              </div>
              
              <div className="flex justify-end">
                <button
                  type="submit"
                  className="w-full sm:w-auto px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50"
                  disabled={loading}
                >
                  {loading ? 'Loading...' : 'Get Records'}
                </button>
              </div>
            </form>
            
            {error && (
              <div className="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {!loading && !error && records.length === 0 && (
              <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-yellow-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M8.485 2.495c.673-1.167 2.357-1.167 3.03 0l6.28 10.875c.673 1.167-.17 2.625-1.516 2.625H3.72c-1.347 0-2.189-1.458-1.515-2.625L8.485 2.495zM10 5a.75.75 0 01.75.75v3.5a.75.75 0 01-1.5 0v-3.5A.75.75 0 0110 5zm0 9a1 1 0 100-2 1 1 0 000 2z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-yellow-700">No records found.</p>
                  </div>
                </div>
              </div>
            )}
            
            {!loading && !error && records.length > 0 && (
              <div className="relative">
                <div className="overflow-x-auto w-full custom-scrollbar">
                  <div className="min-w-max w-full">
                    <table className="min-w-full divide-y divide-gray-300">
                      <thead>
                        <tr>
                          <th className="sticky left-0 z-10 px-2 py-3 text-left text-sm font-semibold text-gray-900 bg-gray-100 w-auto">
                            <span className="whitespace-nowrap">Actions</span>
                          </th>
                        {visibleColumns.map((column) => (
                          <th
                            key={column}
                            className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 bg-gray-100 max-w-[200px]"
                          >
                            {formatFieldName(column)}
                          </th>
                        ))}
                      </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-200 bg-white">
                      {records.map((record, index) => {
                        const recordId = getRecordId(record);
                        const clientId = getClientId(record);
                        
                        return (
                          <tr key={`row-${index}-${recordId}`} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                            <td className="sticky left-0 z-10 px-2 py-3 text-gray-800 w-auto bg-white">
                              <div className="flex flex-row flex-nowrap gap-1 justify-start">
                                <Link
                                  href={`/client?clid=${recordId}`}
                                  className="px-2 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-xs whitespace-nowrap"
                                >
                                  View
                                </Link>
                                <Link
                                  href={`/client-update?clid=${recordId}`}
                                  className="px-2 py-1 bg-blue-100 text-blue-700 rounded hover:bg-blue-200 text-xs whitespace-nowrap"
                                >
                                  Edit
                                </Link>
                                <button
                                  className="px-2 py-1 bg-green-100 text-green-700 rounded hover:bg-green-200 text-xs whitespace-nowrap"
                                  onClick={() => {
                                    if (!clientId) {
                                      console.error('No client ID found for record:', record);
                                      alert('Error: Could not determine client ID');
                                      return;
                                    }
                                    
                                    const clientName = record['Name'] || '';
                                    sessionStorage.setItem('cl-name', clientName);
                                    sessionStorage.setItem('clid', clientId);
                                    window.location.href = `/tasks?clid=${clientId}`;
                                  }}
                                >
                                  Tasks
                                </button>
                              </div>
                            </td>
                            {visibleColumns.map((column) => (
                              <td
                                key={column}
                                className={`px-3 py-4 text-sm text-gray-800 ${column === 'Email' || column === 'Phone' ? 'whitespace-nowrap' : 'break-words'}`}
                              >
                                {formatCellValue(column, record[column])}
                              </td>
                            ))}
                          </tr>
                        );
                      })}
                    </tbody>
                    </table>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
