import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';

export async function POST(request: NextRequest) {
  try {
    const { hostId, databaseName, tableConfig } = await request.json();
    
    if (!hostId || !databaseName || !tableConfig) {
      return NextResponse.json(
        { error: 'Missing required parameters' },
        { status: 400 }
      );
    }
    
    // Path to the databases.ts file
    const databasesFilePath = path.join(process.cwd(), 'lib', 'config', 'databases.ts');
    
    // Read the current file content
    const fileContent = fs.readFileSync(databasesFilePath, 'utf8');
    
    // Find the host section
    const hostRegex = new RegExp(`\\{\\s*hostId:\\s*['"]${hostId}['"]`);
    const hostMatch = fileContent.match(hostRegex);
    
    if (!hostMatch || hostMatch.index === undefined) {
      return NextResponse.json(
        { error: `Host with ID ${hostId} not found` },
        { status: 404 }
      );
    }
    
    // Find the database section within the host
    const hostStartIndex = hostMatch.index;
    const hostSection = fileContent.substring(hostStartIndex);
    
    // Fix: Use databaseName instead of name in the regex
    const databaseRegex = new RegExp(`\\{\\s*databaseName:\\s*['"]${databaseName}['"]`);
    const databaseMatch = hostSection.match(databaseRegex);
    
    if (!databaseMatch || databaseMatch.index === undefined) {
      return NextResponse.json(
        { error: `Database with name ${databaseName} not found in host ${hostId}` },
        { status: 404 }
      );
    }
    
    // Find the tables array
    const databaseStartIndex = hostStartIndex + databaseMatch.index;
    const databaseSection = fileContent.substring(databaseStartIndex);
    
    const tablesMatch = databaseSection.match(/tables\s*:\s*\[/);
    
    if (!tablesMatch || tablesMatch.index === undefined) {
      return NextResponse.json(
        { error: 'Tables array not found' },
        { status: 500 }
      );
    }
    
    // Find the position to insert the new table config
    const tablesStartIndex = databaseStartIndex + tablesMatch.index + tablesMatch[0].length;
    
    // Extract the table name from the TypeScript code
    const tableNameMatch = tableConfig.match(/tableName:\s*['"](.*?)['"]/);
    const tableName = tableNameMatch ? tableNameMatch[1] : 'Unknown';
    
    // Check if the table already exists
    const tableNameRegex = new RegExp(`tableName:\\s*['"]${tableName}['"]`);
    const tableExistsMatch = databaseSection.match(tableNameRegex);
    
    // Prepare the updated content
    let updatedContent;
    let message;
    
    if (tableExistsMatch && tableExistsMatch.index !== undefined) {
      // Find the start and end of the existing table configuration
      const tableStartIndex = databaseStartIndex + tableExistsMatch.index;
      const tableSection = fileContent.substring(tableStartIndex);
      
      // Find the end of the table object by counting braces
      let braceCount = 0;
      let endIndex = 0;
      let foundOpeningBrace = false;
      
      for (let i = 0; i < tableSection.length; i++) {
        const char = tableSection[i];
        
        if (char === '{') {
          braceCount++;
          foundOpeningBrace = true;
        } else if (char === '}') {
          braceCount--;
          if (foundOpeningBrace && braceCount === 0) {
            endIndex = i + 1; // Include the closing brace
            break;
          }
        }
      }
      
      // Find the comma after the table object, if it exists
      let commaIndex = endIndex;
      while (commaIndex < tableSection.length) {
        if (tableSection[commaIndex] === ',') {
          endIndex = commaIndex + 1; // Include the comma
          break;
        } else if (!tableSection[commaIndex].match(/\s/)) {
          // If we find a non-whitespace character that's not a comma, stop looking
          break;
        }
        commaIndex++;
      }
      
      // Replace the existing table configuration
      // Clean up the tableConfig to ensure no extra braces
      const cleanedTableConfig = tableConfig.trim().replace(/^\s*\{\s*\{/, '{');
      
      updatedContent = 
        fileContent.substring(0, tableStartIndex) + 
        cleanedTableConfig + ',' +
        fileContent.substring(tableStartIndex + endIndex);
      
      message = `Table ${tableName} updated in ${databaseName} in host ${hostId}`;
    } else {
      // Insert the new table configuration at the beginning of the tables array
      // Clean up the tableConfig to ensure no extra braces
      const cleanedTableConfig = tableConfig.trim().replace(/^\s*\{\s*\{/, '{');
      
      updatedContent = 
        fileContent.substring(0, tablesStartIndex) + 
        '\n          ' + cleanedTableConfig + ',' +
        fileContent.substring(tablesStartIndex);
      
      message = `Table ${tableName} added to ${databaseName} in host ${hostId}`;
    }
    
    // Write the updated content back to the file
    fs.writeFileSync(databasesFilePath, updatedContent, 'utf8');
    
    return NextResponse.json({ 
      success: true, 
      message: message
    });
    
  } catch (error) {
    console.error('Error updating database config:', error);
    return NextResponse.json(
      { error: `Failed to update database config: ${error instanceof Error ? error.message : String(error)}` },
      { status: 500 }
    );
  }
}
