import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import { hosts, HostConfig, DatabaseConfig, TableConfig } from '@/lib/config/databases';

// Define the path to the custom configurations file
const CONFIG_FILE_PATH = path.join(process.cwd(), 'lib', 'config', 'field-editability.json');

// Define interface for field configuration
interface FieldConfiguration {
  hostId: string;
  databaseName: string;
  tableName: string;
  editableFields: string[];
  readOnlyFields: string[];
  hiddenFields?: string[];
}

// Define the interface for the stored configurations
interface StoredConfigurations {
  configurations: FieldConfiguration[];
}

// Helper function to read the current configurations
function readConfigurations(): StoredConfigurations {
  try {
    // Check if the file exists
    if (fs.existsSync(CONFIG_FILE_PATH)) {
      const fileContent = fs.readFileSync(CONFIG_FILE_PATH, 'utf-8');
      return JSON.parse(fileContent);
    }
    
    // If file doesn't exist, return empty configurations
    return { configurations: [] };
  } catch (error) {
    console.error('Error reading configurations:', error);
    return { configurations: [] };
  }
}

// Helper function to write configurations to file
function writeConfigurations(configurations: StoredConfigurations): boolean {
  try {
    // Ensure the directory exists
    const dirPath = path.dirname(CONFIG_FILE_PATH);
    if (!fs.existsSync(dirPath)) {
      fs.mkdirSync(dirPath, { recursive: true });
    }
    
    // Write the configurations to file
    fs.writeFileSync(CONFIG_FILE_PATH, JSON.stringify(configurations, null, 2), 'utf-8');
    return true;
  } catch (error) {
    console.error('Error writing configurations:', error);
    return false;
  }
}

// Helper function to update the in-memory database configuration
function updateDatabaseConfig(config: FieldConfiguration): boolean {
  try {
    // Find the host
    const hostIndex = hosts.findIndex(host => host.hostId === config.hostId);
    if (hostIndex === -1) return false;
    
    // Find the database
    const dbIndex = hosts[hostIndex].databases.findIndex(db => db.databaseName === config.databaseName);
    if (dbIndex === -1) return false;
    
    // Find the table
    const tableIndex = hosts[hostIndex].databases[dbIndex].tables.findIndex(table => table.tableName === config.tableName);
    if (tableIndex === -1) return false;
    
    // Get the table reference
    const table = hosts[hostIndex].databases[dbIndex].tables[tableIndex];
    
    // Update the table configuration
    table.editableFields = config.editableFields;
    table.readOnlyFields = config.readOnlyFields;
    table.hiddenFields = config.hiddenFields || [];
    
    // Update the hidden property on each field
    if (table.fields && config.hiddenFields) {
      table.fields.forEach(field => {
        // Set hidden flag based on whether the field is in hiddenFields
        field.hidden = config.hiddenFields?.includes(field.fieldName) || false;
      });
    }
    
    return true;
  } catch (error) {
    console.error('Error updating database configuration:', error);
    return false;
  }
}

// POST handler to save field configuration
export async function POST(request: NextRequest) {
  try {
    // Parse the request body
    const config: FieldConfiguration = await request.json();
    
    // Validate required parameters
    if (!config.hostId || !config.databaseName || !config.tableName || !config.editableFields || !config.readOnlyFields) {
      return NextResponse.json(
        { error: 'Missing required parameters' },
        { status: 400 }
      );
    }
    
    // Ensure hiddenFields is an array
    const hiddenFields = config.hiddenFields || [];
    
    // Read current configurations
    const storedConfigs = readConfigurations();
    
    // Check if configuration for this table already exists
    const configIndex = storedConfigs.configurations.findIndex(
      c => c.hostId === config.hostId && 
           c.databaseName === config.databaseName && 
           c.tableName === config.tableName
    );
    
    const newConfig = {
      hostId: config.hostId,
      databaseName: config.databaseName,
      tableName: config.tableName,
      editableFields: config.editableFields,
      readOnlyFields: config.readOnlyFields,
      hiddenFields: hiddenFields
    };
    
    // Update or add the configuration
    if (configIndex !== -1) {
      storedConfigs.configurations[configIndex] = newConfig;
    } else {
      storedConfigs.configurations.push(newConfig);
    }
    
    // Write the updated configurations to file
    const writeSuccess = writeConfigurations(storedConfigs);
    if (!writeSuccess) {
      return NextResponse.json(
        { error: 'Failed to write configuration to file' },
        { status: 500 }
      );
    }
    
    // Update the in-memory database configuration
    const updateSuccess = updateDatabaseConfig(newConfig);
    if (!updateSuccess) {
      return NextResponse.json(
        { error: 'Failed to update in-memory database configuration' },
        { status: 500 }
      );
    }
    
    // Return success response
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Error processing request:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

// GET handler to retrieve field configuration
export async function GET(request: NextRequest) {
  try {
    // Get query parameters
    const url = new URL(request.url);
    const hostId = url.searchParams.get('hostId');
    const databaseName = url.searchParams.get('databaseName');
    const tableName = url.searchParams.get('tableName');
    
    // Read current configurations
    const storedConfigs = readConfigurations();
    
    // If no parameters are provided, return all configurations
    if (!hostId && !databaseName && !tableName) {
      return NextResponse.json(storedConfigs);
    }
    
    // If parameters are provided but not all required ones
    if (!(hostId && databaseName && tableName)) {
      return NextResponse.json(
        { error: 'If filtering, all parameters are required: hostId, databaseName, tableName' },
        { status: 400 }
      );
    }
    
    // Find the configuration for the specified table
    const config = storedConfigs.configurations.find(
      c => c.hostId === hostId && 
           c.databaseName === databaseName && 
           c.tableName === tableName
    );
    
    if (!config) {
      return NextResponse.json(
        { error: 'Configuration not found' },
        { status: 404 }
      );
    }
    
    // Return the configuration
    return NextResponse.json(config);
  } catch (error) {
    console.error('Error processing request:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}
