'use client';

import { useState } from 'react';

type TestResult = {
  status: number;
  headers: Record<string, string>;
  body: any;
};

type ErrorData = {
  message: string;
  details?: any;
};

export default function TestFMConnection() {
  const [result, setResult] = useState<TestResult | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<ErrorData | null>(null);

  const testConnection = async () => {
    setLoading(true);
    setError(null);
    try {
      const response = await fetch('/api/test-connection');
      const data = await response.json();
      setResult(data);
    } catch (err) {
      console.error('Test connection error:', err);
      setError({
        message: err instanceof Error ? err.message : 'An unknown error occurred',
        details: err
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen p-8 bg-gray-50">
      <div className="max-w-5xl mx-auto space-y-6">
        <h1 className="text-3xl font-bold text-gray-900">FileMaker Connection Test</h1>
        
        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <button
            onClick={testConnection}
            disabled={loading}
            className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 disabled:bg-gray-400"
          >
            {loading ? 'Testing...' : 'Test Connection'}
          </button>
        </div>

        {error && (
          <div className="bg-red-50 border-l-4 border-red-400 p-4">
            <h3 className="text-red-700 font-medium">Error: {error.message}</h3>
            {error.details && (
              <pre className="mt-2 p-4 bg-red-100 rounded overflow-auto text-sm">
                {JSON.stringify(error.details, null, 2)}
              </pre>
            )}
          </div>
        )}

        {result && (
          <div className="space-y-4">
            <div className="p-4 bg-blue-50 rounded">
              <span className="font-medium">Status:</span> {result.status}
            </div>
            
            <div className="space-y-2">
              <h3 className="font-medium">Response Headers:</h3>
              <pre className="p-4 bg-gray-50 rounded overflow-auto text-sm">
                {JSON.stringify(result.headers, null, 2)}
              </pre>
            </div>

            <div className="space-y-2">
              <h3 className="font-medium">Response Body:</h3>
              <pre className="p-4 bg-gray-50 rounded overflow-auto text-sm max-h-96">
                {typeof result.body === 'string' 
                  ? result.body 
                  : JSON.stringify(result.body, null, 2)}
              </pre>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
