'use client';

import { useState, useEffect, Suspense, useCallback } from 'react';
import './mobile-styles.css';
import { useRouter, useSearchParams } from 'next/navigation';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import NavBar from '@/components/NavBar';
import { statusOptions, priorityOptions, categoryOptions, noteTypeOptions, topicOptions } from '@/lib/config/field-options';

interface FormField {
  key: string;
  value: any;
  type: string;
  required: boolean;
  displayName: string;
  options?: string[];
  multiline?: boolean;
  readOnly?: boolean;
  step?: string;
  inputmode?: string;
  rows?: number;
}

// Main content component that uses useSearchParams
function NewTaskContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentHostId,
    setCurrentDatabaseName,
    setCurrentTableName
  } = useDatabaseContext();
  
  const [formFields, setFormFields] = useState<FormField[]>([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [saveSuccess, setSaveSuccess] = useState(false);
  const [clientName, setClientName] = useState('');
  
  // Format time in 24-hour format for time input
  const formatTimeForInput = useCallback((date: Date) => {
    const hours = date.getHours().toString().padStart(2, '0');
    const minutes = date.getMinutes().toString().padStart(2, '0');
    return `${hours}:${minutes}`;
  }, []);
  
  // Format time in 12-hour AM/PM format for display
  const formatTimeFor12Hour = useCallback((date: Date) => {
    const hours = date.getHours();
    const minutes = date.getMinutes().toString().padStart(2, '0');
    const ampm = hours >= 12 ? 'PM' : 'AM';
    const hours12 = hours % 12 || 12; // Convert 0 to 12 for 12-hour format
    return `${hours12}:${minutes} ${ampm}`;
  }, []);
  
  // Get parameters from URL
  const hostId = searchParams ? searchParams.get('host') : null;
  const databaseName = searchParams ? searchParams.get('database') : null;
  const tableName = searchParams ? searchParams.get('table') : 'TrackPad';
  const clid = searchParams ? searchParams.get('clid') : null;
  
  // Log the client ID if present
  useEffect(() => {
    if (clid) {
      console.log(`Client ID (clid) detected in URL: ${clid}`);
      // Also log the raw URL search params for debugging
      console.log('Raw URL search params:', window.location.search);
    } else {
      console.warn('No clid parameter found in URL');
    }
  }, [clid]);
  
  // Set the context values from URL parameters if provided
  useEffect(() => {
    if (hostId) setCurrentHostId(hostId);
    if (databaseName) setCurrentDatabaseName(databaseName);
    if (tableName) setCurrentTableName(tableName);
  }, [hostId, databaseName, tableName, setCurrentHostId, setCurrentDatabaseName, setCurrentTableName]);

  // Initialize form fields based on task record structure
  useEffect(() => {
    if (!currentTable) return;

    // Get client name from session storage
    const storedClientName = typeof window !== 'undefined' ? sessionStorage.getItem('cl-name') || '' : '';
    setClientName(storedClientName);

    console.log('Initializing form with clid:', clid);
    console.log('Current URL search params:', window.location.search);

    const fields: FormField[] = [
      { 
        key: 'Message', 
        value: '', 
        type: 'text', 
        required: true,
        displayName: 'Message',
        multiline: true,
        rows: 4
      },
      { 
        key: 'Name', 
        value: storedClientName, 
        type: 'text', 
        required: true,
        displayName: 'Name',
        readOnly: true
      },
      { 
        key: 'pl_Pri', 
        value: 'Low', 
        type: 'select',
        required: false,
        displayName: 'Priority',
        options: [...priorityOptions]
      },
      { 
        key: 'NoteType_c', 
        value: 'GoEasy', 
        type: 'select',
        required: true,
        displayName: 'Type',
        options: [...noteTypeOptions]
      },
      { 
        key: 'Status', 
        value: 'Active', 
        type: 'select',
        required: true,
        displayName: 'Status',
options: [...statusOptions]
      },
      { 
        key: 'StartDate', 
        value: new Date().toISOString().split('T')[0],
        type: 'date',
        required: true,
        displayName: 'Start Date'
      },
      { 
        key: 'StartTime', 
        value: formatTimeFor12Hour(new Date()), // Use 12-hour format with AM/PM
        type: 'time',
        inputmode: 'none', // Prevent mobile keyboard
        required: false,
        displayName: 'Start Time'
      },
      { 
        key: 'EndTime', 
        value: '',
        type: 'time',
        required: false,
        displayName: 'End Time'
      },
      { 
        key: 'pl_Cat', 
        value: 'MyStuff', 
        type: 'select',
        required: false,
        displayName: 'Category',
        options: [...categoryOptions]
      },
      { 
        key: 'pl_Topic', 
        value: 'General', 
        type: 'select',
        required: false,
        displayName: 'Topic',
        options: [...topicOptions]
      },
      { 
        key: 'Name_id', 
        value: clid || '', 
        type: 'number', 
        required: true,
        displayName: 'Name ID',
        readOnly: true
      }
    ];

    console.log('Setting form fields with Name_id:', fields.find(f => f.key === 'Name_id')?.value);
    setFormFields(fields);
    setLoading(false);
  }, [currentTable, clid]); // Add clid to dependencies

  const handleFieldChange = (key: string, value: any) => {
    setFormFields(prevFields => 
      prevFields.map(field => 
        field.key === key ? { ...field, value } : field
      )
    );
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!currentHostId || !currentDatabaseName || !currentTableName) {
      setError('Please select a host, database, and table');
      return;
    }

    // No required fields - all fields are optional

    try {
      setSaving(true);
      setError(null);
      
      // For testing: Include Message, NoteType, Name_id, StartDate, and StartTime fields
      const currentTime = new Date().toTimeString().slice(0, 5); // HH:MM format
      
      // Format time to ensure it has AM/PM if it's a valid time
      const formatTime = (timeStr: string) => {
        if (!timeStr) return '';
        
        // If it already has AM/PM, return as is
        if (timeStr.includes('AM') || timeStr.includes('PM')) {
          return timeStr;
        }
        
        // Try to parse and format the time
        const timeMatch = timeStr.match(/^(\d{1,2}):?(\d{2})?/);
        if (timeMatch) {
          let hours = parseInt(timeMatch[1], 10);
          const minutes = timeMatch[2] ? timeMatch[2].padStart(2, '0') : '00';
          const ampm = hours >= 12 ? 'PM' : 'AM';
          hours = hours % 12;
          hours = hours || 12; // Convert 0 to 12 for 12-hour format
          return `${hours}:${minutes} ${ampm}`;
        }
        return timeStr; // Return as is if we can't parse it
      };
      
      const startTimeValue = formFields.find(f => f.key === 'StartTime')?.value || '';
      const endTimeValue = formFields.find(f => f.key === 'EndTime')?.value || '';
      
      // Get the client ID from the URL parameter or form field
      const nameId = clid || formFields.find(f => f.key === 'Name_id')?.value || '';
      
      const data = {
        Message: formFields.find(f => f.key === 'Message')?.value || '',
        NoteType: formFields.find(f => f.key === 'NoteType_c')?.value || 'GoEasy',
        Name_id: Number(nameId) || 0,
        StartDate: formFields.find(f => f.key === 'StartDate')?.value || new Date().toISOString().split('T')[0],
        StartTime: formatTime(startTimeValue),
        EndTime: formatTime(endTimeValue)
      };
      
      console.log('Creating task with data:', {
        ...data,
        clidFromUrl: clid,
        nameIdFromForm: formFields.find(f => f.key === 'Name_id')?.value
      });
      
      // No validation - submit whatever fields are present

      const requestBody = JSON.stringify(data, null, 2);
      
      // Make logs more visible in browser console
      console.log('%c=== REQUEST BODY ===', 'color: #4CAF50; font-size: 16px; font-weight: bold');
      console.log('%c' + requestBody, 'color: #2196F3;');
      
      console.log('%c=== REQUEST DETAILS ===', 'color: #4CAF50; font-size: 16px; font-weight: bold');
      console.table({
        'Method': 'POST',
        'URL': `/api/odata/${currentDatabaseName}/TrackPad`,
        'Content-Type': 'application/json',
        'X-Host-Id': currentHostId,
        'Accept': 'application/json',
        'OData-Version': '4.0',
        'OData-MaxVersion': '4.0',
        'Body': JSON.parse(requestBody)
      });
      
      // Add a debug button to re-run the request
      console.log('%cDebug: Click to copy request', 'background: #2196F3; color: white; padding: 5px; border-radius: 3px;', {
        request: {
          url: `/api/odata/${currentDatabaseName}/TrackPad`,
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0'
          },
          body: JSON.parse(requestBody)
        },
        copyToClipboard: () => {
          navigator.clipboard.writeText(JSON.stringify({
            url: `/api/odata/${currentDatabaseName}/TrackPad`,
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-Host-Id': currentHostId,
              'Accept': 'application/json',
              'OData-Version': '4.0',
              'OData-MaxVersion': '4.0'
            },
            body: JSON.parse(requestBody)
          }, null, 2));
          console.log('%cRequest copied to clipboard!', 'color: #4CAF50;');
        }
      });
      
      const response = await fetch(`/api/odata/${currentDatabaseName}/TrackPad`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Host-Id': currentHostId,
          'Accept': 'application/json',
          'OData-Version': '4.0',
          'OData-MaxVersion': '4.0'
        },
        body: JSON.stringify(data)
      });

      if (!response.ok) {
        let errorData: { error?: { message?: string } } = {};
        try {
          errorData = await response.json();
          console.error('Error Response:', {
            status: response.status,
            statusText: response.statusText,
            error: errorData,
            request: {
              url: `/api/odata/${currentDatabaseName}/TrackPad`,
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
                'X-Host-Id': currentHostId,
                'Accept': 'application/json',
                'OData-Version': '4.0',
                'OData-MaxVersion': '4.0'
              },
              body: data
            }
          });
        } catch (e) {
          console.error('Failed to parse error response:', e);
        }
        throw new Error(errorData.error?.message || `Failed to create task (${response.status} ${response.statusText})`);
      }

      setSaveSuccess(true);
      
      // Show success message
      console.log('%c✅ Task created successfully!', 'color: #4CAF50; font-size: 16px; font-weight: bold;');
      
      // Redirect back to tasks page with client ID after a short delay
      setTimeout(() => {
        if (clid) {
          console.log(`Redirecting to /tasks?clid=${clid}`);
          router.push(`/tasks?clid=${clid}`);
        } else {
          console.log('No clid found, redirecting to /tasks');
          router.push('/tasks');
        }
      }, 1500);
      
      // Clear the form after successful submission
      setFormFields(prevFields => 
        prevFields.map(field => ({
          ...field,
          value: field.required ? field.value : '' // Keep required fields, clear others
        }))
      );
      
    } catch (err) {
      console.error('Error creating task:', err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred');
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="bg-white rounded-lg shadow-lg p-6">
            <div className="flex justify-center items-center h-64">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      <div className="max-w-4xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200">
            <h1 className="text-2xl font-semibold text-gray-900">
              Create New Task
            </h1>
          </div>
          
          <div className="p-6">
            {error && (
              <div className="mb-6 bg-red-50 border-l-4 border-red-400 p-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {saveSuccess && (
              <div className="mb-6 bg-green-50 border-l-4 border-green-400 p-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-green-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-green-700">Task created successfully! Redirecting to tasks list...</p>
                  </div>
                </div>
              </div>
            )}
            
            <form onSubmit={handleSubmit} className="space-y-6 database-content">
              <div className="bg-white overflow-hidden sm:rounded-lg">
                <div className="px-6 py-5 space-y-6">
                  <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                    {formFields
                      .map((field) => (
                      <div key={field.key} className={field.readOnly ? 'opacity-75' : ''}>
                        <label htmlFor={field.key} className="block text-sm font-medium text-gray-700">
                          {field.displayName}
                          {field.required && <span className="ml-1 text-red-500">*</span>}
                        </label>
                        <div className="mt-1">
                          {field.readOnly ? (
                            <div className="mt-1 block w-full py-2 px-3 border border-gray-300 bg-gray-50 rounded-md shadow-sm text-black sm:text-sm">
                              {field.value}
                            </div>
                          ) : field.type === 'select' ? (
                            <select
                              id={field.key}
                              name={field.key}
                              value={field.value || ''}
                              onChange={(e) => handleFieldChange(field.key, e.target.value)}
                              className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                              disabled={field.readOnly || saving}
                            >
                              {field.options?.map((option) => (
                                <option key={option} value={option}>
                                  {option}
                                </option>
                              ))}
                            </select>
                          ) : field.type === 'date' ? (
                            <input
                              type="date"
                              id={field.key}
                              name={field.key}
                              value={field.value || ''}
                              onChange={(e) => handleFieldChange(field.key, e.target.value)}
                              className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                              disabled={field.readOnly || saving}
                            />
                          ) : field.type === 'time' ? (
                            <div className="relative">
                              <input
                                type="time"
                                id={field.key}
                                name={field.key}
                                value={(() => {
                                  // Convert stored time (e.g., "2:30 PM") back to 24h format for the input
                                  const timeStr = field.value || '';
                                  if (!timeStr) return '';
                                  
                                  const [time, period] = timeStr.split(' ');
                                  if (!time || !period) return '';
                                  
                                  let [hours, minutes] = time.split(':');
                                  if (period === 'PM' && hours !== '12') {
                                    hours = String(parseInt(hours, 10) + 12);
                                  } else if (period === 'AM' && hours === '12') {
                                    hours = '00';
                                  }
                                  return `${hours.padStart(2, '0')}:${minutes || '00'}`;
                                })()}
                                onChange={(e) => {
                                  const timeValue = e.target.value;
                                  if (timeValue) {
                                    const [hours, minutes] = timeValue.split(':');
                                    const hoursNum = parseInt(hours, 10);
                                    const ampm = hoursNum >= 12 ? 'PM' : 'AM';
                                    const hours12 = hoursNum % 12 || 12;
                                    const formattedTime = `${hours12}:${minutes || '00'} ${ampm}`;
                                    handleFieldChange(field.key, formattedTime);
                                  } else {
                                    handleFieldChange(field.key, '');
                                  }
                                }}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                                disabled={field.readOnly || saving}
                              />
                              <div 
                                className="absolute inset-y-0 right-0 flex items-center pr-3 cursor-pointer"
                                onClick={(e) => {
                                  e.preventDefault();
                                  const input = document.getElementById(field.key) as HTMLInputElement;
                                  if (input) input.showPicker();
                                }}
                              >
                                <svg className="h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
                                </svg>
                              </div>
                            </div>
                          ) : field.multiline ? (
                            <div className="relative">
                              <textarea
                                id={field.key}
                                name={field.key}
                                rows={field.rows || 4}
                                value={field.value || ''}
                                onChange={(e) => handleFieldChange(field.key, e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                                disabled={field.readOnly || saving}
                                style={{
                                  minHeight: '6rem',
                                  lineHeight: '1.5',
                                  padding: '0.5rem',
                                }}
                              />
                            </div>
                          ) : (
                            <input
                              type={field.type}
                              id={field.key}
                              name={field.key}
                              value={field.value || ''}
                              onChange={(e) => handleFieldChange(field.key, e.target.value)}
                              className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                              disabled={field.readOnly || saving}
                            />
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
              
              <div className="px-6 py-4 bg-gray-50 border-t border-gray-200">
                <div className="flex justify-between items-center">
                  <div className="text-sm text-gray-500">
                    Fields marked with <span className="text-red-500">*</span> are required
                  </div>
                  <div className="flex space-x-3">
                    <Link
                      href={`/tasks-all?host=${currentHostId}&database=${currentDatabaseName}&table=${currentTableName}`}
                      className="inline-flex items-center px-4 py-2 border border-gray-300 shadow-sm text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                    >
                      Cancel
                    </Link>
                    <button
                      type="submit"
                      disabled={saving}
                      className="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {saving ? 'Saving...' : 'Save Task'}
                    </button>
                  </div>
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function NewTaskPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">New Task</h1>
              <p>Loading task form...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <NewTaskContent />
    </Suspense>
  );
}
